<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
declare(strict_types=1);

namespace app\index\common\taglib;

use think\template\TagLib;
use think\Exception;

class Rycms extends TagLib
{
    // +------------------------------------------------------------------
    // | 标签定义
    // +------------------------------------------------------------------
    protected $tags = [
        'lists' => ['attr' => 'idname,cid,limit,order,type,index,model,as', 'close' => 1],
        'sing'  => ['attr' => 'model,field,id', 'close' => 0]
    ];

    // +------------------------------------------------------------------
    // | 单条数据查询标签 
    // +------------------------------------------------------------------
    public function tagSing(array $tag): string
    {
        $model = $this->safeModelClass($tag['model'] ?? '');
        $field = $this->safeFieldName($tag['field'] ?? '');
        $id    = $this->parseId($tag['id'] ?? 0);

        return <<<PHP
        <?php
        try {
            \$instance = {$model}::where('id', {$id})->find();
            echo \$instance->{$field} ?? '';
        } catch (\Throwable \$e) {
            echo '';
        }
        ?>
        PHP;
    }

    // +------------------------------------------------------------------
    // | 列表数据查询标签 
    // +------------------------------------------------------------------

    public function tagLists(array $tag, string $content): string
    {
        $modelName = $tag['model'] ?? 'Article';
        $model     = $this->safeModelClass($modelName);
        $cateField = $this->safeFieldName($tag['idname'] ?? 'cateid');
        $limit     = min((int)($tag['limit'] ?? 10), 100);
        $order     = $this->safeOrder($tag['order'] ?? 'create_time DESC');
        $cids      = $this->parseIds($tag['cid'] ?? '0');
        $varName   = $this->safeFieldName($tag['as'] ?? 'v');

        $queryCode = "<?php 
        \$list = {$model}::where('status', 1)"
            . $this->buildWhere([
                'type'      => $tag['type'] ?? null,
                'index'     => $tag['index'] ?? null,
                $cateField  => $cids ?: null
            ])
            . "->order('{$order}')"
            . "->limit({$limit})"
            . "->select(); 
            
        // 自动为Article模型的每条数据添加URL字段
        if ('{$modelName}' === 'Article') {
            foreach (\$list as &\$item) {
                if (!empty(\$item['link'])) {
                    \$item['url'] = \$item['link'];
                } else {
                    \$salt = 'royal';
                    \$encrypted = base64_encode(\$item['id'] . '|' . md5(\$item['id'] . \$salt));
                    \$encryptedId = rtrim(strtr(\$encrypted, '+/', '-_'), '=');
                    \$item['url'] = '/article/' . \$encryptedId . '.html';
                }
            }
            unset(\$item);
        }
        ?>";

        $safeContent = preg_replace_callback(
            '/\{\$v\.(.*?)\}/',
            fn($m) => '<?= htmlspecialchars($v[\'' . $this->safeFieldName($m[1]) . '\'] ?? \'\', ENT_QUOTES) ?>',
            $content
        );

        return "{$queryCode}<?php foreach(\$list as \${$varName}): ?>{$safeContent}<?php endforeach; ?>";
    }
    // +------------------------------------------------------------------
    // | 安全模型类验证 
    // +------------------------------------------------------------------
    private function safeModelClass(string $name): string
    {
        $name = ucfirst($name);
        if (!preg_match('/^[A-Z][a-zA-Z]+$/', $name)) {
            throw new Exception("Invalid model name: {$name}");
        }
        $class = '\\app\\index\\model\\' . $name;
        if (!class_exists($class)) {
            throw new Exception("Model class [{$class}] not exists");
        }
        return $class;
    }

    // +------------------------------------------------------------------
    // | 安全字段名验证 
    // +------------------------------------------------------------------
    private function safeFieldName(string $field): string
    {
        return preg_replace('/[^\w]/', '', $field);
    }

    // +------------------------------------------------------------------
    // | 解析ID值 
    // +------------------------------------------------------------------
    private function parseId($val): string
    {
        $cleanVal = is_numeric($val)
            ? (int)$val
            : "'" . addslashes((string)$val) . "'";
        return $this->safeFieldName((string)$cleanVal);
    }

    // +------------------------------------------------------------------
    // | 安全排序处理 - 修复版
    // +------------------------------------------------------------------
    private function safeOrder(string $order): string
    {
        // 允许的排序字段（扩展更多常用字段）
        $allowFields = [
            'sort',
            'create_time',
            'update_time',
            'id',
            'click',
            'price',
            'star',
        ];

        // 处理多个排序条件，如 "click DESC, create_time DESC"
        $orderParts = explode(',', $order);
        $safeOrders = [];

        foreach ($orderParts as $part) {
            $part = trim($part);
            $subParts = explode(' ', preg_replace('/\s+/', ' ', $part));

            $field = $this->safeFieldName($subParts[0]);
            $direction = isset($subParts[1]) && in_array(strtoupper($subParts[1]), ['ASC', 'DESC'])
                ? $subParts[1]
                : 'ASC';

            // 如果字段在白名单中，使用原字段；否则使用默认排序
            if (in_array($field, $allowFields)) {
                $safeOrders[] = "{$field} {$direction}";
            } else {
                // 对于不在白名单的字段，使用默认排序而不是直接拒绝
                $safeOrders[] = "create_time {$direction}";
            }
        }

        return empty($safeOrders) ? 'create_time DESC' : implode(', ', $safeOrders);
    }

    // +------------------------------------------------------------------
    // | 解析分类ID集合 
    // +------------------------------------------------------------------
    private function parseIds($ids): array
    {
        if (is_array($ids)) {
            return array_filter($ids, 'is_numeric');
        }
        return array_filter(
            explode(',', (string)$ids),
            fn($id) => is_numeric(trim($id))
        );
    }

    // +------------------------------------------------------------------
    // | 构建WHERE条件语句 
    // +------------------------------------------------------------------
    private function buildWhere(array $conditions): string
    {
        $code = '';
        foreach ($conditions as $field => $value) {
            if (null === $value) continue;

            $safeField = $this->safeFieldName((string)$field);
            if (is_array($value)) {
                $cleanValues = array_map('intval', $value);
                $code .= sprintf('->whereIn("%s", %s)', $safeField, var_export($cleanValues, true));
            } else {
                $cleanValue = is_numeric($value) ? (int)$value : "'" . addslashes((string)$value) . "'";
                $code .= sprintf('->where("%s", %s)', $safeField, $cleanValue);
            }
        }
        return $code;
    }
}
