<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------


declare(strict_types=1);

namespace app\index\controller;

use app\index\controller\Common;
use app\index\model\Article as ArticleModel;
use think\exception\HttpException;
use think\facade\Cache;
use think\facade\Lang;

class Article extends Common
{
    //-----------------------------------------------------------------
    // 文章详情页
    //-----------------------------------------------------------------
    public function index()
    {
        try {
            $article = $this->getValidArticle();
            $this->updateClickCount($article->id);

            return view($this->buildTemplatePath($article->bltCate), $this->prepareViewData($article));
        } catch (HttpException $e) {
            return $this->errorView($e->getMessage());
        }
    }

    //-----------------------------------------------------------------
    // 获取验证过的文章数据
    //-----------------------------------------------------------------
    private function getValidArticle(): ArticleModel
    {
        $id = $this->decryptId(request()->param('encryptedId'));
        
        if ($id <= 0) {
            throw new HttpException(404, Lang::get('article_data_not_exists'));
        }

        $article = Cache::remember("article_{$id}", fn() => 
            ArticleModel::with(['bltCate'])->find($id), 3600
        );

        if (!$article || $article->status != 1 || $article->type == 5) {
            throw new HttpException(404, Lang::get('article_not_exists_or_offline'));
        }

        return $article;
    }

    //-----------------------------------------------------------------
    // 准备视图数据
    //-----------------------------------------------------------------
    private function prepareViewData(ArticleModel $article): array
    {
        return [
            'Arts'       => $article,
            'Cates'      => $article->bltCate,
            'topcid'     => $this->getRootCategoryId($article->cateid) ?: $article->cateid,
            'cateid'     => $article->cateid,
            'SonNavMenu' => $this->buildSonNav($article->cateid),
            'PosBread'   => $this->generateBreadcrumbs($article->cateid),
            'PrevArts'   => $this->getArticleNeighbors($article)['prev'],
            'NextArts'   => $this->getArticleNeighbors($article)['next'],
            'ArtRes'     => $this->getRelatedProducts($article),
            'images'     => $this->processImages($article->images),
            'tags'       => $this->processTags($article->tag),
        ];
    }

    //-----------------------------------------------------------------
    // 更新点击量
    //-----------------------------------------------------------------
    private function updateClickCount(int $id): void
    {
        ArticleModel::where('id', $id)->inc('click', rand(5, 10))->update();
    }

    //-----------------------------------------------------------------
    // 获取相邻文章
    //-----------------------------------------------------------------
    private function getArticleNeighbors(ArticleModel $article): array
    {
        return Cache::remember("article_neighbors_{$article->id}", function() use ($article) {
            $ids = ArticleModel::where('cateid', $article->cateid)
                ->where('status', 1)
                ->where('type', '<>', 5)
                ->order('sort ASC, create_time DESC')
                ->column('id');

            $currentKey = array_search($article->id, $ids);

            return [
                'prev' => $this->getNeighborArticle($ids, $currentKey - 1),
                'next' => $this->getNeighborArticle($ids, $currentKey + 1)
            ];
        }, 3600);
    }

    //-----------------------------------------------------------------
    // 获取相邻文章详情
    //-----------------------------------------------------------------
    private function getNeighborArticle(array $ids, int $offset): ?ArticleModel
    {
        return isset($ids[$offset]) 
            ? Cache::remember("article_{$ids[$offset]}", fn() => 
                ArticleModel::find($ids[$offset]), 3600)
            : null;
    }

    //-----------------------------------------------------------------
    // 获取相关产品
    //-----------------------------------------------------------------
    private function getRelatedProducts(ArticleModel $article, int $limit = 6)
    {
        return Cache::remember("related_products_{$article->id}", function() use ($article, $limit) {
            return ArticleModel::where('cateid', $article->cateid)
                ->where('id', '<>', $article->id)
                ->where('status', 1)
                ->where('type', '<>', 5)
                ->field('id,cateid,type,sort,create_time,price,star,click,stitle,title,keywords,description,thumb,content,link')
                ->order('sort ASC, create_time DESC')
                ->limit($limit)
                ->select()
                ->each(function ($product) {
                    $product->tags = $this->processTags($product->tag);
                    $product->url = $this->generateArticleUrl($product);
                });
        }, 3600);
    }

    //-----------------------------------------------------------------
    // 构建模板路径
    //-----------------------------------------------------------------
    private function buildTemplatePath($cate): string
    {
        return $this->getCurrentTheme() . '/' . basename($cate->article_tmp);
    }
}