<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

namespace app\index\controller;

use app\BaseController;
use app\index\model\{Conf, Cate, Link, Article as ArticleModel};
use think\facade\View;
use think\facade\Lang; 
use think\exception\HttpResponseException;

class Common extends BaseController
{
    //-----------------------------------------------------------------
    // 初始化（自动加载公共资源）
    //-----------------------------------------------------------------
    public function initialize(): void
    {
        parent::initialize();
 
        if (!$this->checkSiteStatus()) {
            $this->throwMaintenanceException();
        }
        $this->loadCommonResources();
    }
 
    //-----------------------------------------------------------------
    // 站点状态检查
    //-----------------------------------------------------------------
    protected function checkSiteStatus(): bool
    {
        return Conf::where('ename', 'siteon')->value('value') === '1';
    }

    //-----------------------------------------------------------------
    // 抛出维护异常（统一异常处理）
    //-----------------------------------------------------------------
    protected function throwMaintenanceException(): void
    {
        $response = request()->isAjax()
            ? json(['code' => 503, 'msg' => Lang::get('common_website_msg')])
            : view('error/index_off', ['message' => Lang::get('common_website_msg')])->code(503);

        throw new HttpResponseException($response);
    }

    //-----------------------------------------------------------------
    // 获取主题路径 
    //-----------------------------------------------------------------
    protected function getCurrentTheme(string $templateFile = ''): string
    {
        $basePath = app()->getRootPath() . config('view.view_dir_name') . '/default';

        if (empty($templateFile)) {
            return $basePath;
        } 
        return "{$basePath}/{$templateFile}";
    }

    //-----------------------------------------------------------------
    // 加载公共模板变量
    //-----------------------------------------------------------------
    protected function loadCommonResources(): void
    {
        View::assign([
            'CurLang'  => Lang::getLangSet(),
            'ConfRes'  => $this->getMergedConfig(),
            'NavMenu'  => $this->getTopCategories(),
            'LinkRes'  => Link::order('sort')->select()->toArray()
        ]);
    }

    //-----------------------------------------------------------------
    // 获取合并后的多语言配置
    //----------------------------------------------------------------
    protected function getMergedConfig(): array
    {
        return Conf::cache(3600)->column('value', 'ename');
    }

    //-----------------------------------------------------------------
    // 获取带子级的顶级分类 
    //-----------------------------------------------------------------
    protected function getTopCategories(): array
    {
        return Cate::with(['children' => fn($q) => $q->where('status', 1)->order('sort ASC')])
            ->field('title,ename,id,pid,type,link')
            ->where(['status' => 1, 'navmenu' => 1])
            ->where('pid', 0)
            ->order('sort ASC')
            ->cache(3600)
            ->select()
            ->toArray();
    }

    //-----------------------------------------------------------------
    // 获取栏目所有子级ID（包含自身） 
    //-----------------------------------------------------------------
    public function getAllChildIds(int $cid): array
    {
        $cacheKey = 'category_children_' . $cid;

        if ($allIds = cache($cacheKey)) {
            return $allIds;
        }

        try {
            $allIds = Cate::field('id')
                ->whereRaw("FIND_IN_SET(pid, getAncestry({$cid}))")
                ->column('id');
        } catch (\Throwable $e) {
            $allIds = [];
            foreach (Cate::where('pid', $cid)->column('id') as $id) {
                $allIds = array_merge($allIds, [$id], $this->getAllChildIds($id));
            }
        }

        cache($cacheKey, $allIds, 3600);
        return $allIds;
    }

    //-----------------------------------------------------------------
    // 获取根分类ID（向上追溯） 
    //-----------------------------------------------------------------
    public function getRootCategoryId(int $cid): int
    {
        while (($cate = Cate::find($cid)) && $cate->pid !== 0) {
            $cid = $cate->pid;
        }
        return $cid;
    }

    //-----------------------------------------------------------------
    // 构建子导航（智能返回子级或同级） 
    //-----------------------------------------------------------------
    protected function buildSonNav(int $cateId): array
    {
        

        $cate = Cate::findOrFail($cateId);
        $query = Cate::order('sort')->where('status', 1)->field('id,pid,title,ename,type,link');

        $hasChildren = Cate::where('pid', $cateId)->limit(1)->count() > 0;
        $navData = $hasChildren
            ? $query->where('pid', $cateId)->select()->toArray()
            : $query->where('pid', $cate->pid)->select()->toArray();
 
        return $navData;
    }

    //-----------------------------------------------------------------
    // 生成面包屑导航（包含当前分类及所有父级） 
    //-----------------------------------------------------------------
    public function generateBreadcrumbs(int $cid): array
    {
        $crumbs = [];
        $current = Cate::find($cid);

        if (!$current) return $crumbs;

        $crumbs[] = $current->toArray();

        while ($current->pid !== 0) {
            $current = Cate::find($current->pid);
            if (!$current) break;
            array_unshift($crumbs, $current->toArray());
        }

        return $crumbs;
    }

    //-----------------------------------------------------------------
    // 处理图片数据
    //-----------------------------------------------------------------
    protected function processImages(?string $imagesString): array
    {
        return empty($imagesString) 
            ? [] 
            : array_filter(array_map('trim', explode(',', $imagesString)));
    }

    //-----------------------------------------------------------------
    // 处理标签数据（带颜色）
    //-----------------------------------------------------------------
    protected function processTags(?string $tagString): array
    {
        if (empty($tagString) || $tagString === ',') return [];

        $tags = array_filter(array_map('trim', explode(',', trim($tagString, ', '))));
        if (empty($tags)) return [];

        $colors = ['blue', 'green', 'orange', 'red', 'purple'];
        
        return array_map(fn($index, $tag) => [
            'name'  => $tag,
            'color' => $colors[$index % count($colors)]
        ], array_keys($tags), $tags);
    }

    //-----------------------------------------------------------------
    // 处理文章标签（简单格式）
    //-----------------------------------------------------------------
    protected function processArticleTags(?string $tagString): array
    {
        if (empty($tagString)) return [];

        $tags = array_filter(array_map('trim', explode(',', $tagString)));
        
        return array_map(fn($tag) => ['name' => $tag], $tags);
    }

    //-----------------------------------------------------------------
    // 生成文章链接
    //-----------------------------------------------------------------
    protected function generateArticleUrl($article): string
    {
        return !empty($article->link) 
            ? $article->link 
            : "/article/" . $this->encryptId($article->id) . ".html";
    }

    //-----------------------------------------------------------------
    // 错误页面视图
    //-----------------------------------------------------------------
    protected function errorView(string $message)
    {
        return view('error/exception_jump', [
            'jump_msg'  => $message,
            'jump_url'  => url('/'),
            'wait_time' => 5
        ]);
    }

    //──────────────────────────────────────────────────
    // ID加密解密方法
    //──────────────────────────────────────────────────

    //-----------------------------------------------------------------
    // 加密ID
    //-----------------------------------------------------------------
    public function encryptId($id): string
    {
        $salt = 'royal';
        $encrypted = base64_encode($id . '|' . md5($id . $salt));
        return rtrim(strtr($encrypted, '+/', '-_'), '=');
    }

    //-----------------------------------------------------------------
    // 解密ID
    //-----------------------------------------------------------------
    public function decryptId($encryptedId)
    {
        $salt = 'royal';
        $decoded = base64_decode(str_pad(strtr($encryptedId, '-_', '+/'), strlen($encryptedId) % 4, '=', STR_PAD_RIGHT));

        list($id, $hash) = explode('|', $decoded);
        return md5($id . $salt) === $hash ? (int)$id : false;
    }
}