<?php
namespace app\index\controller;

use app\index\controller\Common;
use think\facade\Db;
use think\facade\Validate;
use think\exception\ValidateException;

class Message extends Common
{
    public function index()
    {
        if (!$this->request->isPost()) {
            return json(['code' => 403, 'msg' => lang('illegal_request')]);
        }

        try {
            $rawData = $this->request->post();
            $filteredData = $this->sanitizeInput($rawData);

            $validate = Validate::rule([
                'name|' . lang('name') => 'require|max:25',
                'phone|' . lang('phone') => 'require|mobile',
                'email|' . lang('email') => 'email',
                'content|' . lang('content') => 'require|max:500'
            ]);

            if (!$validate->check($filteredData)) {
                return json(['code' => 400, 'msg' => $validate->getError()]);
            }

            //-----------------------------------------------------------------
            // 修复：构建符合数据库表结构的字段
            //-----------------------------------------------------------------
            $dbData = [
                'name' => $filteredData['name'],
                'phone' => $filteredData['phone'],
                'email' => $filteredData['email'],
                'content' => $filteredData['content'],
                'create_time' => time(),
                'update_time' => time(), // 添加更新时间
                'ip' => $this->request->ip(),
                'user_agent' => substr($this->request->server('HTTP_USER_AGENT'), 0, 255),
                'status' => 0, // 注意：数据库默认是0-待处理，不是1
                'form_type' => 1, // 数据库是tinyint类型，不是字符串
                // 其他字段使用默认值
                'tel' => '',
                'num' => '',
                'card_id' => '',
                'address' => '',
                'passport' => '',
                'wechat' => '',
                'verify_code' => '',
                'city' => '',
                'area' => ''
            ];

            //-----------------------------------------------------------------
            // 修复：使用正确的表名并添加调试
            //-----------------------------------------------------------------
            $result = Db::name('form')->insert($dbData);
            
            // 调试信息
            if (!$result) {
                $error = Db::getError();
                // 临时返回详细错误信息用于调试
                return json([
                    'code' => 500, 
                    'msg' => '插入失败',
                    'debug' => [
                        'error' => $error,
                        'sql' => Db::getLastSql(),
                        'data' => $dbData
                    ]
                ]);
            }

            return json(['code' => 200, 'msg' => lang('message_submit_success')]);
            
        } catch (\Exception $e) {
            // 返回详细异常信息用于调试
            return json([
                'code' => 500, 
                'msg' => lang('system_error'),
                'debug' => $e->getMessage()
            ]);
        }
    }

    private function sanitizeInput(array $input): array
    {
        return [
            'name'    => htmlspecialchars(trim($input['name'] ?? ''), ENT_QUOTES, 'UTF-8'),
            'phone'   => preg_replace('/[^\d]/', '', $input['phone'] ?? ''),
            'email'   => filter_var($input['email'] ?? '', FILTER_SANITIZE_EMAIL),
            'content' => htmlspecialchars(trim($input['content'] ?? ''), ENT_QUOTES, 'UTF-8'),
        ];
    }
}