<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

namespace app\royaladmin\common;

use think\facade\Db;
use think\facade\Session;

/**
 * 权限认证类
 */
class Auth
{
    // -------------------------------------------------------------------------
    // 类常量与属性
    // -------------------------------------------------------------------------

    /** @var array 默认配置 */
    protected $_config = [
        'auth_on'           => true,
        'auth_type'         => 1,
        'auth_group'        => 'auth_group',
        'auth_group_access' => 'auth_group_access',
        'auth_rule'         => 'auth_rule',
        'auth_user'         => 'auser',
    ];

    /** @var array 用户信息缓存 */
    protected static $userCache = [];

    // -------------------------------------------------------------------------
    // 初始化方法
    // -------------------------------------------------------------------------

    //-----------------------------------------------------------------
    // 构造方法
    //-----------------------------------------------------------------
    public function __construct()
    {
        $this->initializeConfig();
    }

    //-----------------------------------------------------------------
    // 初始化配置
    //-----------------------------------------------------------------
    protected function initializeConfig()
    {
        $prefix = config('database.prefix');
        
        $this->_config = array_map(function($value) use ($prefix) {
            return in_array($value, ['auth_group', 'auth_rule', 'auth_user', 'auth_group_access']) 
                ? $prefix . $value 
                : $value;
        }, $this->_config);

        if ($customConfig = config('auth.auth_config')) {
            $this->_config = array_merge($this->_config, $customConfig);
        }
    }

    // -------------------------------------------------------------------------
    // 核心权限验证方法
    // -------------------------------------------------------------------------

    //-----------------------------------------------------------------
    // 权限验证
    //-----------------------------------------------------------------
    public function check($name, $uid, $type = 1, $mode = 'url', $relation = 'or')
    {
        if (!$this->_config['auth_on']) return true;

        $authList = $this->getAuthList($uid, $type);
        $nameList = $this->normalizeName($name);

        return $this->validate($authList, $nameList, $mode, $relation);
    }

    // -------------------------------------------------------------------------
    // 辅助方法
    // -------------------------------------------------------------------------

    //-----------------------------------------------------------------
    // 标准化规则名称
    //-----------------------------------------------------------------
    protected function normalizeName($name)
    {
        if (!is_array($name)) {
            $name = explode(',', strtolower(str_replace(' ', '', $name)));
        }
        return array_map('strtolower', $name);
    }

    //-----------------------------------------------------------------
    // 执行验证逻辑
    //-----------------------------------------------------------------
    protected function validate($authList, $nameList, $mode, $relation)
    {
        $matched = [];
        $request = array_change_key_case($_REQUEST, CASE_LOWER);

        foreach ($authList as $rule) {
            if ($mode === 'url' && strpos($rule, '?') !== false) {
                list($base, $query) = explode('?', $rule, 2);
                if ($this->matchUrlRule($base, $query, $nameList, $request)) {
                    $matched[] = $base;
                }
            } elseif (in_array($rule, $nameList)) {
                $matched[] = $rule;
            }
        }

        return $relation === 'or' 
            ? !empty($matched) 
            : empty(array_diff($nameList, $matched));
    }

    //-----------------------------------------------------------------
    // 匹配URL规则
    //-----------------------------------------------------------------
    protected function matchUrlRule($base, $query, $nameList, $request)
    {
        parse_str($query, $ruleParams);
        $intersect = array_intersect_assoc($request, $ruleParams);
        
        return in_array(strtolower($base), $nameList) && $intersect == $ruleParams;
    }

    // -------------------------------------------------------------------------
    // 用户组与权限获取
    // -------------------------------------------------------------------------

    //-----------------------------------------------------------------
    // 获取用户组
    //-----------------------------------------------------------------
    public function getGroups($uid)
    {
        static $groupCache = [];

        if (!isset($groupCache[$uid])) {
            $query = Db::table($this->_config['auth_group_access'])
                ->alias('a')
                ->join([$this->_config['auth_group'] => 'g'], 'a.group_id = g.id')
                ->where('a.uid', $uid)
                ->where('g.status', 1)
                ->field('uid,group_id,title,rules');

            $groupCache[$uid] = $query->select()->toArray() ?: [];
        }

        return $groupCache[$uid];
    }

    //-----------------------------------------------------------------
    // 获取权限列表
    //-----------------------------------------------------------------
    protected function getAuthList($uid, $type)
    {
        $cacheKey = $uid . '_' . $type;

        if (isset($_authList[$cacheKey])) {
            return $_authList[$cacheKey];
        }

        if ($this->useSessionCache($cacheKey)) {
            return Session::get('_auth_list_' . $cacheKey, []);
        }

        $ruleIds = $this->getUserRuleIds($uid);
        $authList = $this->fetchValidRules($ruleIds, $type, $uid);

        return $this->cacheAuthList($cacheKey, $authList);
    }

    // -------------------------------------------------------------------------
    // 数据库操作相关
    // -------------------------------------------------------------------------

    //-----------------------------------------------------------------
    // 获取用户规则ID集合
    //-----------------------------------------------------------------
    protected function getUserRuleIds($uid)
    {
        $groups = $this->getGroups($uid);
        $ruleIds = [];

        foreach ($groups as $group) {
            $ruleIds = array_merge($ruleIds, explode(',', trim($group['rules'], ',')));
        }

        return array_unique(array_filter($ruleIds));
    }

    //-----------------------------------------------------------------
    // 获取有效规则列表
    //-----------------------------------------------------------------
    protected function fetchValidRules($ruleIds, $type, $uid)
    {
        if (empty($ruleIds)) return [];

        $query = Db::table($this->_config['auth_rule'])
            ->where('id', 'in', $ruleIds)
            ->where('type', $type)
            ->where('status', 1)
            ->field('condition,name');

        return $query->select()->map(function($rule) use ($uid) {
            return $this->evaluateRuleCondition($rule, $uid) ? strtolower($rule['name']) : null;
        })->filter()->toArray();
    }

    //-----------------------------------------------------------------
    // 评估规则条件
    //-----------------------------------------------------------------
    protected function evaluateRuleCondition($rule, $uid)
    {
        if (empty($rule['condition'])) return true;

        $user = $this->getUserInfo($uid);
        $condition = preg_replace_callback('/\{(\w+)\}/', fn($m) => ($user[$m[1]] ?? 'null'), $rule['condition']);

        try {
            return eval("return ($condition);");
        } catch (\Throwable $e) {
            return false;
        }
    }

    // -------------------------------------------------------------------------
    // 缓存处理
    // -------------------------------------------------------------------------

    //-----------------------------------------------------------------
    // 检查是否使用会话缓存
    //-----------------------------------------------------------------
    protected function useSessionCache($key)
    {
        return $this->_config['auth_type'] === 2 && Session::has('_auth_list_' . $key);
    }

    //-----------------------------------------------------------------
    // 缓存权限列表
    //-----------------------------------------------------------------
    protected function cacheAuthList($key, $data)
    {
        static $_authList = [];
        $data = array_unique($data);

        if ($this->_config['auth_type'] === 2) {
            Session::set('_auth_list_' . $key, $data);
        }

        return $_authList[$key] = $data;
    }

    // -------------------------------------------------------------------------
    // 用户信息处理
    // -------------------------------------------------------------------------

    //-----------------------------------------------------------------
    // 获取用户信息
    //-----------------------------------------------------------------
    protected function getUserInfo($uid)
    {
        if (!isset(self::$userCache[$uid])) {
            self::$userCache[$uid] = Db::table($this->_config['auth_user'])
                ->where('id', $uid)
                ->find() ?: [];
        }

        return self::$userCache[$uid];
    }
}