<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

namespace app\royaladmin\common;

use PDO;
use PDOException;
use RuntimeException;

class Baksql
{
    // -------------------------------------------------------------------------
    // 配置参数和连接句柄
    // -------------------------------------------------------------------------
    private array $config;
    private ?PDO $handler;
    private float $startTime;

    // -------------------------------------------------------------------------
    // 初始化数据库连接和配置
    // -------------------------------------------------------------------------
    public function __construct(array $config)
    {
        $this->initializeConfig($config);
        $this->startTime = microtime(true);
        $this->establishConnection();
    }

    // -------------------------------------------------------------------------
    // 配置参数初始化
    // -------------------------------------------------------------------------
    private function initializeConfig(array $config): void
    {
        $mysql = $config['connections']['mysql'];
        $backupDir = root_path().'runtime/databak/sqlbak/';

        $this->config = [
            'dsn'        => "{$mysql['type']}:host={$mysql['hostname']};port={$mysql['hostport']};dbname={$mysql['database']}",
            'username'   => $mysql['username'],
            'password'   => $mysql['password'],
            'charset'    => $mysql['charset'],
            'backup_dir' => $backupDir,
            'backup_file' => sprintf('Royal_%s_%s.sql', 
                date('YmdHis'), 
                md5(hash('crc32', microtime(true)))
            )
        ];

        $this->validateDirectory($backupDir);
    }

    // -------------------------------------------------------------------------
    // 建立数据库连接
    // -------------------------------------------------------------------------
    private function establishConnection(): void
    {
        try {
            $this->handler = new PDO(
                $this->config['dsn'],
                $this->config['username'],
                $this->config['password'],
                [
                    PDO::MYSQL_ATTR_INIT_COMMAND => "SET NAMES {$this->config['charset']}",
                    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
                    PDO::ATTR_EMULATE_PREPARES => false
                ]
            );
        } catch (PDOException $e) {
            throw new RuntimeException("数据库连接失败: ".$e->getMessage());
        }
    }

    // -------------------------------------------------------------------------
    // 执行数据库备份（仅备份数据表）
    // -------------------------------------------------------------------------
    public function executeBackup(): string
    {
        try {
            $tables = $this->getDatabaseTables();
            if (empty($tables)) {
                throw new RuntimeException('没有找到可备份的表');
            }

            $backupContent = $this->generateBackupContent($tables);
            $filePath = $this->config['backup_dir'].$this->config['backup_file'];

            if (file_put_contents($filePath, $backupContent) === false) {
                throw new RuntimeException('备份文件写入失败');
            }

            return "备份成功，共备份 " . count($tables) . " 个表，耗时 {$this->getExecTime()}";

        } catch (PDOException $e) {
            throw new RuntimeException("备份失败: ".$e->getMessage());
        }
    }

    // -------------------------------------------------------------------------
    // 获取数据库表列表（仅数据表，排除视图）
    // -------------------------------------------------------------------------
    private function getDatabaseTables(): array
    {
        // 只获取基础表，排除视图
        $stmt = $this->handler->query("SHOW FULL TABLES WHERE Table_type = 'BASE TABLE'");
        return $stmt->fetchAll(PDO::FETCH_COLUMN);
    }

    // -------------------------------------------------------------------------
    // 生成备份文件内容
    // -------------------------------------------------------------------------
    private function generateBackupContent(array $tables): string
    {
        $content = [
            "-- MySQL Data Backup\n",
            "-- Server: {$this->config['dsn']}\n",
            "-- Date: ".date('Y-m-d H:i:s')."\n",
            "-- Tables: " . count($tables) . "\n",
            "SET NAMES {$this->config['charset']};\n",
            "SET FOREIGN_KEY_CHECKS=0;\n\n"
        ];

        foreach ($tables as $table) {
            $content[] = $this->getTableSchema($table);
            $content[] = $this->getTableData($table);
        }

        // 恢复外键检查
        $content[] = "SET FOREIGN_KEY_CHECKS=1;\n";
        $content[] = "-- Backup completed at " . date('Y-m-d H:i:s') . " --\n";

        return implode('', $content);
    }

    // -------------------------------------------------------------------------
    // 获取表结构定义
    // -------------------------------------------------------------------------
    private function getTableSchema(string $table): string
    {
        $stmt = $this->handler->query("SHOW CREATE TABLE `{$table}`");
        $schema = $stmt->fetch(PDO::FETCH_NUM)[1];
        return "-- Table structure for `{$table}`\nDROP TABLE IF EXISTS `{$table}`;\n{$schema};\n\n";
    }

    // -------------------------------------------------------------------------
    // 获取表数据（改进的INSERT语句生成）
    // -------------------------------------------------------------------------
    private function getTableData(string $table): string
    {
        try {
            $stmt = $this->handler->query("SELECT * FROM `{$table}`");
            $columns = $this->getTableColumns($table);
            $data = [];
            $rowCount = 0;

            while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
                $values = [];
                foreach ($row as $value) {
                    if ($value === null) {
                        $values[] = 'NULL';
                    } else {
                        // 使用更安全的转义方式
                        $values[] = $this->handler->quote($value);
                    }
                }
                $data[] = "INSERT INTO `{$table}` ({$columns}) VALUES (".implode(',', $values).");";
                $rowCount++;
                
                // 每1000行输出一次进度信息
                if ($rowCount % 1000 === 0) {
                    $data[] = "-- {$table} data progress: {$rowCount} rows --\n";
                }
            }

            if ($data) {
                return "-- Data for `{$table}` ({$rowCount} rows) --\n".implode("\n", $data)."\n\n";
            } else {
                return "-- No data in table `{$table}` --\n\n";
            }

        } catch (PDOException $e) {
            // 如果表数据读取失败，记录错误但继续备份其他表
            return "-- Error reading data from table `{$table}`: " . $e->getMessage() . " --\n\n";
        }
    }

    // -------------------------------------------------------------------------
    // 获取表字段列表
    // -------------------------------------------------------------------------
    private function getTableColumns(string $table): string
    {
        $stmt = $this->handler->query("SHOW COLUMNS FROM `{$table}`");
        $columns = $stmt->fetchAll(PDO::FETCH_COLUMN);
        return '`'.implode('`,`', $columns).'`';
    }

    // -------------------------------------------------------------------------
    // 还原数据库备份（完全修复的事务管理）
    // -------------------------------------------------------------------------
    public function restoreBackup(string $filename): string
    {
        $filePath = $this->validateBackupFile($filename);
        
        try {
            // 不使用事务，因为SQL文件可能包含DDL语句（CREATE TABLE等）
            // DDL语句在MySQL中会自动提交事务，导致事务管理复杂
            $this->executeSqlFileWithoutTransaction($filePath);
            
            return "还原成功，耗时 {$this->getExecTime()}";
            
        } catch (\Exception $e) {
            throw new RuntimeException("还原失败: ".$e->getMessage());
        }
    }

    // -------------------------------------------------------------------------
    // 无事务执行SQL文件（推荐的还原方式）
    // -------------------------------------------------------------------------
    private function executeSqlFileWithoutTransaction(string $filePath): void
    {
        $handle = fopen($filePath, 'r');
        if (!$handle) {
            throw new RuntimeException('无法打开备份文件');
        }

        $sql = '';
        $lineNumber = 0;
        $successCount = 0;
        $errorCount = 0;

        try {
            while (!feof($handle)) {
                $line = fgets($handle);
                $lineNumber++;

                // 跳过注释行和空行
                if (preg_match('/^\s*(--|#)/', $line) || trim($line) === '') {
                    continue;
                }

                $sql .= $line;

                // 如果以分号结尾，表示一个完整的SQL语句
                if (substr(rtrim($line), -1) === ';') {
                    // 移除可能的多余空格和分号
                    $sql = trim($sql);
                    
                    if (!empty($sql) && $sql !== ';') {
                        try {
                            // 执行SQL语句
                            $this->handler->exec($sql);
                            $successCount++;
                        } catch (PDOException $e) {
                            $errorCount++;
                            // 记录错误但继续执行（可选：可以在这里决定是否停止）
                            error_log("SQL执行警告 (第{$lineNumber}行): " . $e->getMessage() . "\nSQL: " . $sql);
                            
                            // 如果是外键检查错误，可以尝试禁用外键检查
                            if (strpos($e->getMessage(), 'foreign key constraint') !== false) {
                                $this->handler->exec("SET FOREIGN_KEY_CHECKS=0");
                                // 重新尝试执行
                                try {
                                    $this->handler->exec($sql);
                                    $successCount++;
                                    $errorCount--;
                                } catch (PDOException $e2) {
                                    // 如果还是失败，抛出异常停止执行
                                    throw new RuntimeException("关键SQL执行失败 (第{$lineNumber}行): " . $e2->getMessage() . "\nSQL: " . $sql);
                                }
                                $this->handler->exec("SET FOREIGN_KEY_CHECKS=1");
                            } else {
                                // 对于非关键错误，继续执行
                                continue;
                            }
                        }
                    }
                    
                    // 重置SQL变量
                    $sql = '';
                }
            }

            // 处理文件末尾可能没有分号的最后一条SQL
            if (!empty(trim($sql))) {
                try {
                    $this->handler->exec(trim($sql));
                    $successCount++;
                } catch (PDOException $e) {
                    $errorCount++;
                    error_log("SQL执行警告 (文件末尾): " . $e->getMessage() . "\nSQL: " . $sql);
                }
            }

            // 如果有错误，记录日志
            if ($errorCount > 0) {
                error_log("SQL文件执行完成：成功 {$successCount} 条，失败 {$errorCount} 条");
            }

        } catch (\Exception $e) {
            throw new RuntimeException("SQL执行错误 (第{$lineNumber}行): " . $e->getMessage() . "\nSQL: " . $sql);
        } finally {
            fclose($handle);
        }
    }

    // -------------------------------------------------------------------------
    // 备选方案：使用事务的还原方法（如果需要原子性）
    // -------------------------------------------------------------------------
    public function restoreBackupWithTransaction(string $filename): string
    {
        $filePath = $this->validateBackupFile($filename);
        
        // 先检查是否支持事务
        if (!$this->isTransactionSupported()) {
            return $this->restoreBackup($filename); // 回退到无事务版本
        }
        
        try {
            // 开始事务
            $this->handler->beginTransaction();
            
            // 执行SQL文件
            $this->executeSqlFileInTransaction($filePath);
            
            // 提交事务
            $this->handler->commit();
            
            return "还原成功，耗时 {$this->getExecTime()}";
            
        } catch (\Exception $e) {
            // 安全地回滚事务
            $this->safeRollback();
            throw new RuntimeException("还原失败: ".$e->getMessage());
        }
    }

    // -------------------------------------------------------------------------
    // 在事务中执行SQL文件
    // -------------------------------------------------------------------------
    private function executeSqlFileInTransaction(string $filePath): void
    {
        $handle = fopen($filePath, 'r');
        if (!$handle) {
            throw new RuntimeException('无法打开备份文件');
        }

        $sql = '';
        $lineNumber = 0;

        try {
            while (!feof($handle)) {
                $line = fgets($handle);
                $lineNumber++;

                // 跳过注释行和空行
                if (preg_match('/^\s*(--|#)/', $line) || trim($line) === '') {
                    continue;
                }

                $sql .= $line;

                // 如果以分号结尾，表示一个完整的SQL语句
                if (substr(rtrim($line), -1) === ';') {
                    $sql = trim($sql);
                    
                    if (!empty($sql) && $sql !== ';') {
                        $this->handler->exec($sql);
                    }
                    
                    $sql = '';
                }
            }

            // 处理文件末尾可能没有分号的最后一条SQL
            if (!empty(trim($sql))) {
                $this->handler->exec(trim($sql));
            }

        } catch (\Exception $e) {
            throw new RuntimeException("SQL执行错误 (第{$lineNumber}行): " . $e->getMessage() . "\nSQL: " . $sql);
        } finally {
            fclose($handle);
        }
    }

    // -------------------------------------------------------------------------
    // 安全回滚方法
    // -------------------------------------------------------------------------
    private function safeRollback(): void
    {
        try {
            if ($this->handler->inTransaction()) {
                $this->handler->rollBack();
            }
        } catch (PDOException $e) {
            // 忽略回滚错误，只记录日志
            error_log("回滚事务时发生错误: " . $e->getMessage());
        }
    }

    // -------------------------------------------------------------------------
    // 检查是否支持事务
    // -------------------------------------------------------------------------
    private function isTransactionSupported(): bool
    {
        try {
            // 检查数据库引擎是否支持事务
            $stmt = $this->handler->query("SELECT @@autocommit");
            $autocommit = $stmt->fetch(PDO::FETCH_COLUMN);
            
            // 检查是否有活跃的事务（不应该有）
            if ($this->handler->inTransaction()) {
                return false;
            }
            
            return true;
        } catch (\Exception $e) {
            return false;
        }
    }

    // -------------------------------------------------------------------------
    // 下载备份文件
    // -------------------------------------------------------------------------
    public function downloadBackup(string $filename): void
    {
        $filePath = $this->validateBackupFile($filename);
        
        if (!headers_sent()) {
            header('Content-Description: File Transfer');
            header('Content-Type: application/octet-stream');
            header('Content-Length: '.filesize($filePath));
            header('Content-Disposition: attachment; filename='.basename($filePath));
            
            readfile($filePath);
            exit;
        } else {
            throw new RuntimeException('HTTP头信息已发送，无法下载文件');
        }
    }

    // -------------------------------------------------------------------------
    // 删除备份文件
    // -------------------------------------------------------------------------
    public function deleteBackup(string $filename): string
    {
        $filePath = $this->validateBackupFile($filename);
        
        if (!unlink($filePath)) {
            throw new RuntimeException('删除备份文件失败');
        }
        
        return "文件 {$filename} 已删除";
    }

    // -------------------------------------------------------------------------
    // 获取备份文件列表
    // -------------------------------------------------------------------------
    public function getFileList(): array
    {
        $files = [];
        $dir = new \DirectoryIterator($this->config['backup_dir']);

        foreach ($dir as $file) {
            if ($file->isFile() && $file->getExtension() === 'sql') {
                $files[] = [
                    'name' => $file->getFilename(),
                    'time' => date('Y-m-d H:i:s', $file->getMTime()),
                    'size' => $this->formatSize($file->getSize())
                ];
            }
        }

        // 按时间倒序排列，最新的在前面
        usort($files, fn($a, $b) => $b['time'] <=> $a['time']);
        return $files;
    }

    // -------------------------------------------------------------------------
    // 安全验证备份文件
    // -------------------------------------------------------------------------
    private function validateBackupFile(string $filename): string
    {
        $cleanName = $this->sanitizeFilename($filename);
        $filePath = $this->config['backup_dir'].$cleanName;

        if (!file_exists($filePath)) {
            throw new RuntimeException('备份文件不存在: ' . $filename);
        }

        if (!is_readable($filePath)) {
            throw new RuntimeException('备份文件不可读: ' . $filename);
        }

        return $filePath;
    }

    // -------------------------------------------------------------------------
    // 文件名安全过滤
    // -------------------------------------------------------------------------
    public function sanitizeFilename(string $filename): string
    {
        // 只允许字母、数字、下划线、连字符和点号
        $cleanName = preg_replace('/[^a-zA-Z0-9\-_.]/', '', $filename);
        if ($cleanName !== $filename) {
            throw new RuntimeException('检测到非法文件名');
        }
        
        // 防止目录遍历攻击
        if (strpos($cleanName, '..') !== false) {
            throw new RuntimeException('检测到非法文件名');
        }
        
        return $cleanName;
    }

    // -------------------------------------------------------------------------
    // 辅助方法：格式化文件大小
    // -------------------------------------------------------------------------
    private function formatSize(int $bytes): string
    {
        $units = ['B', 'KB', 'MB', 'GB'];
        $index = 0;
        
        while ($bytes >= 1024 && $index < 3) {
            $bytes /= 1024;
            $index++;
        }
        
        return round($bytes, 2).$units[$index];
    }

    // -------------------------------------------------------------------------
    // 辅助方法：计算执行时间
    // -------------------------------------------------------------------------
    private function getExecTime(): string
    {
        return round(microtime(true) - $this->startTime, 2).'秒';
    }

    // -------------------------------------------------------------------------
    // 辅助方法：验证备份目录
    // -------------------------------------------------------------------------
    private function validateDirectory(string $path): void
    {
        if (!is_dir($path) && !mkdir($path, 0755, true)) {
            throw new RuntimeException('无法创建备份目录: ' . $path);
        }

        if (!is_writable($path)) {
            throw new RuntimeException('备份目录不可写: ' . $path);
        }
    }

    // -------------------------------------------------------------------------
    // 析构函数：正确关闭数据库连接
    // -------------------------------------------------------------------------
    public function __destruct()
    {
        if (isset($this->handler)) {
            unset($this->handler);
        }
    }
}