<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

namespace app\royaladmin\controller;

use app\royaladmin\controller\RyBase;
use think\facade\Request;
use app\royaladmin\model\RyAd as RyAdModel;
use app\royaladmin\validate\RyAd as RyAdValidate;
use app\royaladmin\model\RyAdpos;
use app\royaladmin\model\RyCate;
use app\royaladmin\model\RyArticle;
use app\royaladmin\fields\form\RyAdFields;
use app\royaladmin\fields\list\RyAdList;
use Throwable;

// -----------------------------------------------------------------------------
// 广告管理控制器
// 负责广告的增删改查及展示功能
// -----------------------------------------------------------------------------
class RyAd extends RyBase
{
    // -------------------------------------------------------------------------
    // 类常量与属性
    // -------------------------------------------------------------------------
    protected $modelClass    = RyAdModel::class;    // 数据模型类
    protected $validateClass = RyAdValidate::class; // 数据验证类 

    // -------------------------------------------------------------------------
    // 表单处理方法
    // -------------------------------------------------------------------------
    public function form(?int $id = null)
    {
        // 非AJAX请求渲染视图
        if (!Request::isAjax()) {
            $model = $id ? RyAdModel::find($id) : null; // 查找模型
            $currentData = $model ? $model->getData() : []; // 获取数据

            // 时间字段处理
            if ($model && !empty($currentData['end_time'])) {
                $currentData['end_time'] = $model->end_time; // 格式化时间
            }

            $pageTitle = $id ? lang('edit_ad') : lang('add_ad'); // 页面标题
            $currentParentId = Request::param('adposid/d', 0); // 当前广告位ID

            // 字段选项配置
            $fieldsOptions = [
                'adposid' => RyAdpos::field('id,title')->select()->toArray(), // 广告位选项
                'jump_cate' => RyCate::field('id,title')->select()->toArray(), // 跳转栏目选项
                'jump_article' => RyArticle::field('id,title')->select()->toArray() // 跳转文章选项
            ];

            $formTabs = RyAdFields::getFormTabs(); // 获取表单配置
            $this->assignOptionsForFields($formTabs, $fieldsOptions); // 动态赋值选项

            return parent::renderFormView($formTabs, $currentData, $pageTitle, $currentParentId); // 调用基类渲染
        }

        // AJAX请求处理数据提交
        try {
            $data = Request::param(); // 获取请求数据

            // 记录当前用户ID
            $data['adminid'] = session('auser_id') ?? 2; // 设置管理员ID

            // 到期时间不为空时格式化时间
            if (!empty($data['end_time'])) {
                $data['end_time'] = strtotime($data['end_time']); // 转换为时间戳
            }

            // 调用基类处理表单提交
            $result = parent::handleFormSubmit($data, $id);

            // 记录操作日志
            $adTitle = $data['title'] ?? ($id ? "ID:{$id}" : lang('new_ad'));
            $actionType = $id ? lang('edit') : lang('add');
            $this->logFormAction("{$actionType}" . lang('ad') . "《{$adTitle}》");

            // 返回操作结果
            return json([
                'code' => 200,
                'msg' => "{$actionType}" . lang('ad_success'),
                'url' => url('Ad/lst', ['adposid' => $data['adposid'] ?? 0])->build(), // 返回URL
                'data' => $result['data'] ?? []
            ]);
        } catch (Throwable $e) {
            return json([
                'code' => $e->getCode() ?: 500, // 错误代码
                'msg'  => $e->getMessage(), // 错误消息
            ]);
        }
    }

    // -------------------------------------------------------------------------
    // 广告列表页面
    // -------------------------------------------------------------------------
    public function lst()
    {
        // 获取列表配置并渲染视图
        $config = RyAdList::getListConfig();
        return view('common/list', array_merge($config, $config['viewParams']));
    }

    // -------------------------------------------------------------------------
    // 获取广告数据接口
    // -------------------------------------------------------------------------
    public function getAdData()
    {
        return $this->getCommonData([
            'searchFields' => [
                'title' => 's',   // 标题搜索（字符串匹配）
                'status' => 'd',  // 状态搜索（数字匹配）
                'adposid' => 'd'  // 广告位搜索（数字匹配）
            ],
            'with' => 'adpos',    // 关联广告位模型
            'field' => 'id,title,stitle,thumb,create_time,adposid,status,sort', // 查询字段
            'order' => 'sort asc, create_time DESC' // 排序规则
        ]);
    }
}