<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

namespace app\royaladmin\controller;

use app\royaladmin\controller\RyBase;
use think\facade\Log;
use think\facade\Request;
use app\royaladmin\model\RyAdpos as RyAdposModel;
use app\royaladmin\validate\RyAdpos as RyAdposValidate;
use app\royaladmin\fields\form\RyAdposFields;
use app\royaladmin\fields\list\RyAdposList;
use Throwable;

// -----------------------------------------------------------------------------
// 广告位管理控制器
// 负责广告位的增删改查及关联广告检查
// -----------------------------------------------------------------------------
class RyAdpos extends RyBase
{
    // -------------------------------------------------------------------------
    // 类常量与属性
    // -------------------------------------------------------------------------
    protected $modelClass    = RyAdposModel::class;    // 数据模型类
    protected $validateClass = RyAdposValidate::class; // 数据验证类

    // -------------------------------------------------------------------------
    // 表单处理方法
    // -------------------------------------------------------------------------
    public function form(?int $id = null)
    {
        // 非AJAX请求渲染视图
        if (!Request::isAjax()) {
            $model = $id ? RyAdposModel::find($id) : null; // 查找模型
            $currentData = $model ? $model->getData() : []; // 获取数据
            $pageTitle = $id ? lang('edit_adpos') : lang('add_adpos'); // 页面标题

            $formTabs = RyAdposFields::getFormTabs(); // 获取表单配置

            return parent::renderFormView( // 调用基类渲染方法
                $formTabs,
                $currentData,
                $pageTitle,
                0 // 当前父级ID
            );
        }

        // AJAX请求处理数据提交
        try {
            $data = Request::param(); // 获取请求数据
            
            // 调用基类处理表单提交
            $result = parent::handleFormSubmit($data, $id);

            // 记录操作日志
            $actionType = $id ? lang('edit') : lang('add');
            $logContent = "{$actionType}" . lang('adpos') . "【{$data['title']}】";
            $this->logFormAction($logContent);

            // 返回操作结果
            return json($result + [
                'url' => url('adpos/lst')->build(), // 返回URL
            ]);
        } catch (Throwable $e) {
            return json([
                'code' => $e->getCode() ?: 500, // 错误代码
                'msg' => $e->getMessage(), // 错误消息
            ]);
        }
    }

    // -------------------------------------------------------------------------
    // 广告位列表页面
    // -------------------------------------------------------------------------
    public function lst()
    {
        // 获取列表配置并渲染视图
        $config = RyAdposList::getListConfig();
        return view('common/list', array_merge($config, $config['viewParams']));
    }

    // -------------------------------------------------------------------------
    // 获取广告位数据接口
    // -------------------------------------------------------------------------
    public function getAdposData()
    {
        return $this->getCommonData([
            'withCount' => ['ads'], // 关联广告计数
            'order' => 'sort,id DESC', // 排序规则
        ]);
    }

    // -------------------------------------------------------------------------
    // 删除广告位方法
    // -------------------------------------------------------------------------
    public function del($ids)
    {
        try {
            // 参数规范化处理
            $id = (int)$ids; // 转换为整数

            if ($id <= 0) {
                return ry_error(lang('invalid_delete_params')); // 无效参数
            }

            // 获取广告位信息（包含关联广告计数）
            $adpos = RyAdposModel::withCount('ads')->find($id);

            if (!$adpos) {
                return ry_error(lang('adpos_not_exists')); // 广告位不存在
            }

            // 检查关联广告
            if ($adpos->ads_count > 0) {
                return ry_error(lang('cannot_delete_adpos_with_ads', [$adpos->title, $adpos->ads_count])); // 存在关联广告
            }

            // 执行删除操作
            $result = $adpos->delete();

            if (!$result) {
                return ry_error(lang('delete_adpos_failed', [$adpos->title])); // 删除失败
            }

            // 记录操作日志
            $this->logFormAction(lang('delete_adpos_log', [$adpos->title]));

            return ry_success(lang('delete_adpos_success', [$adpos->title])); // 删除成功
        } catch (Throwable $e) {
            Log::error('广告位删除失败：' . $e->getMessage() . ' ID:' . $id); // 记录错误日志
            return ry_error(lang('delete_failed') . (config('app.app_debug') ? $e->getMessage() : lang('system_error'))); // 返回错误
        }
    }
}