<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn 
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

namespace app\royaladmin\controller;

use app\royaladmin\controller\RyBase;
use think\facade\Db;
use think\facade\Log;
use think\facade\Request;
use app\royaladmin\model\RyCate as RyCateModel;
use app\royaladmin\model\RyArticle as RyArticleModel;
use app\royaladmin\model\RyTag as RyTagModel;
use app\royaladmin\validate\RyArticle as RyArticleValidate;
use app\royaladmin\fields\form\RyArticleFields;
use app\royaladmin\fields\list\RyArticleList;
use Throwable;

// -----------------------------------------------------------------------------
// 文档管理控制器
// 负责文章的增删改查及批量操作功能
// -----------------------------------------------------------------------------
class RyArticle extends RyBase
{
    // -------------------------------------------------------------------------
    // 类常量与属性
    // -------------------------------------------------------------------------
    protected $modelClass = RyArticleModel::class;     // 数据模型类
    protected $validateClass = RyArticleValidate::class; // 数据验证类

    // -------------------------------------------------------------------------
    // 文档列表页面
    // -------------------------------------------------------------------------
    public function lst()
    {
        // 获取列表配置并渲染视图
        $config = RyArticleList::getListConfig();
        return view('common/list', array_merge($config, $config['viewParams']));
    }

    // -------------------------------------------------------------------------
    // 获取文章数据接口
    // -------------------------------------------------------------------------
    public function getArtData()
    {
        // 先获取基础数据
        $jsonResponse = $this->getCommonData([
            'searchFields' => ['title' => 's', 'status' => 'd', 'cateid' => 'd'], // 搜索字段配置
            'field' => 'id,title,thumb,create_time,update_time,cateid,is_index,status,sort,type', // 查询字段
            'order' => 'sort asc, create_time DESC' // 排序规则
        ]);

        // 将Json响应转换为数组进行处理
        $result = json_decode($jsonResponse->getContent(), true);

        // 为文章数据添加加密ID字段
        if (isset($result['data']) && is_array($result['data'])) {
            $result['data'] = array_map(function ($item) {
                $item['encrypted_id'] = $this->encryptId($item['id']); // 加密ID
                return $item;
            }, $result['data']);
        }

        // 返回处理后的数据
        return json($result);
    }

    // -------------------------------------------------------------------------
    // 表单处理方法
    // -------------------------------------------------------------------------
    public function form(?int $id = null)
    {
        // 非AJAX请求渲染视图
        if (!Request::isAjax()) {
            $model = $id ? RyArticleModel::find($id) : null; // 查找模型
            $currentData = $model ? $model->getData() : []; // 获取数据
    
            // 处理时间字段 - 将时间戳转换为日期格式用于表单显示
            if (isset($currentData['create_time']) && is_numeric($currentData['create_time'])) {
                $currentData['create_time'] = date('Y-m-d', $currentData['create_time']);
            }
            // 处理多选字段（转换为数组）
            foreach (['images', 'catesub', 'tag'] as $field) {
                $currentData[$field] = isset($currentData[$field]) && is_string($currentData[$field]) && !empty($currentData[$field])
                    ? explode(',', $currentData[$field]) // 字符串转数组
                    : [];
            }

            // 配置表单字段选项
            $formTabs = RyArticleFields::getFormTabs(); // 获取表单配置
            $this->assignOptionsForFields($formTabs, [
                'cateid' => RyCateModel::getTreeData(true),  // 父级栏目选项
                'catesub' => RyCateModel::getTreeData(false), // 子栏目选项
                'tag' => RyTagModel::getTagOptions() // 标签选项
            ]);

            return parent::renderFormView( // 调用基类渲染方法
                $formTabs,
                $currentData,
                $id ? lang('edit_article') : lang('add_article'), // 页面标题
                Request::param('cateid/d', 0) // 当前分类ID
            );
        }

        // AJAX请求处理表单提交
        try {
            $data = Request::param(); // 获取请求数据

            // 使用模型处理表单数据
            $model = new RyArticleModel();
            $processed = $model->processFormData($data, $id);
            $processed['adminid'] = session('auser_id') ?? 2; // 设置管理员ID

            // 调用基类提交处理
            $result = parent::handleFormSubmit($processed, $id);

            // 记录操作日志
            $actionType = $id ? lang('edit') : lang('add');
            $this->logFormAction("{$actionType}" . lang('article') . "《{$data['title']}》");

            return json($result + [
                'url' => url('article/lst', ['cateid' => $processed['cateid'] ?? 0])->build(), // 返回URL
            ]);
        } catch (Throwable $e) {
            return json([
                'code' => $e->getCode() ?: 500, // 错误代码
                'msg'  => $e->getMessage(), // 错误消息
            ]);
        }
    }

    // -------------------------------------------------------------------------
    // 批量移动文章方法
    // -------------------------------------------------------------------------
    public function moveAll()
    {
        try {
            $ids = array_filter((array) Request::param('ids/a'), 'is_numeric'); // 过滤有效ID
            $moveId = (int) Request::param('cateid', 0); // 目标分类ID

            if (empty($ids)) {
                return ry_error(lang('select_articles_first')); // 未选择文章
            }
            if ($moveId < 0) {
                return ry_error(lang('invalid_parameters')); // 参数错误
            }

            // 验证目标分类（排除移动到根分类的情况）
            if ($moveId !== 0) {
                $cateExists = RyCateModel::where('id', $moveId)->count();
                if (!$cateExists) {
                    return ry_error(lang('target_category_not_exists')); // 分类不存在
                }
            }

            // 批量更新文章分类
            $count = RyArticleModel::whereIn('id', $ids)
                ->update(['cateid' => $moveId]);

            // 记录操作日志
            $this->logFormAction(lang('batch_move_articles_log', [$count]));

            // 返回操作结果
            return $count > 0
                ? ry_success(lang('batch_move_success', [$count]), ['count' => $count])
                : ry_error(lang('no_articles_found'));
        } catch (Throwable $e) {
            Log::error("批量移动异常：{$e->getMessage()} IDS:" . implode(',', $ids ?? [])); // 记录错误
            return ry_error(config('app.app_debug')
                ? lang('system_exception') . $e->getMessage()
                : lang('operation_failed_retry'));
        }
    }

    // -------------------------------------------------------------------------
    // 删除文章方法
    // -------------------------------------------------------------------------
    public function del()
    {
        try {
            // 使用基类方法标准化ID参数
            $ids = $this->normalizeIds(Request::param('ids'));

            if (empty($ids)) {
                return ry_error(lang('select_data_first')); // 未选择数据
            }

            if (count($ids) > 50) {
                return ry_error(lang('max_delete_limit')); // 超出删除限制
            }

            Db::startTrans(); // 开始事务
            $count = RyArticleModel::whereIn('id', $ids)->delete(); // 批量删除

            if ($count === 0) {
                Db::rollback(); // 回滚事务
                return ry_error(lang('no_data_found')); // 未找到数据
            }

            // 记录操作日志
            $this->logFormAction(lang('delete_articles_log', [$count]));
            Db::commit(); // 提交事务

            return ry_success(lang('delete_articles_success', [$count]));
        } catch (Throwable $e) {
            Db::rollback(); // 回滚事务
            Log::error("文档删除失败: {$e->getMessage()} IDS:" . implode(',', $ids ?? [])); // 记录错误

            return json([
                'code' => $e->getCode() ?: 500, // 错误代码
                'msg' => config('app.app_debug') ? $e->getMessage() : lang('operation_failed_check_log') // 错误消息
            ]);
        }
    }

    // -------------------------------------------------------------------------
    // 私有辅助方法
    // -------------------------------------------------------------------------

    /** ID加密方法 - 使用ID+盐进行加密，生成URL安全的字符串 */
    protected function encryptId($id): string
    {
        $salt = 'royal'; // 加密盐值
        $encrypted = base64_encode($id . '|' . md5($id . $salt)); // 加密处理
        return rtrim(strtr($encrypted, '+/', '-_'), '='); // URL安全编码
    }

    /** ID解密方法 - 验证加密字符串并返回原始ID */
    protected function decryptId($encryptedId)
    {
        $salt = 'royal'; // 解密盐值
        $decoded = base64_decode(str_pad(strtr($encryptedId, '-_', '+/'), strlen($encryptedId) % 4, '=', STR_PAD_RIGHT)); // 解码
        list($id, $hash) = explode('|', $decoded); // 分割ID和哈希

        if (md5($id . $salt) === $hash) { // 验证哈希
            return (int)$id; // 返回解密ID
        }

        return false; // 验证失败
    }
}