<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

namespace app\royaladmin\controller;

use app\royaladmin\controller\RyBase;
use think\facade\Db;
use think\facade\Log;
use think\facade\Request;
use app\royaladmin\model\RyAuthRule as RyAuthRuleModel;
use app\royaladmin\validate\RyAuthRule as RyAuthRuleValidate;
use app\royaladmin\fields\form\RyAuthRuleFields;
use app\royaladmin\fields\list\RyRuleList;
use app\royaladmin\model\RyActionLog;
use Throwable;

// -----------------------------------------------------------------------------
// 权限规则管理控制器
// 负责权限规则的增删改查及树形结构管理
// -----------------------------------------------------------------------------
class RyAuthRule extends RyBase
{
    // -------------------------------------------------------------------------
    // 类常量与属性
    // -------------------------------------------------------------------------
    protected $modelClass    = RyAuthRuleModel::class;     // 数据模型类
    protected $validateClass = RyAuthRuleValidate::class; // 数据验证类

    // -------------------------------------------------------------------------
    // 获取权限规则数据接口
    // -------------------------------------------------------------------------
    public function getRuleData()
    {
        $treeData = RyAuthRuleModel::getTreeData(false); // 获取树形数据（不添加前缀）
        
        return json([
            'code' => 0,
            'msg'  => 'success',
            'data' => $treeData, // 返回数据
        ]);
    }

    // -------------------------------------------------------------------------
    // 权限规则列表页面
    // -------------------------------------------------------------------------
    public function lst()
    {
        $config = RyRuleList::getListConfig(); // 获取列表配置
        
        // 确保配置中有树形表格设置
        $config['isTreeTable'] = true;   // 标记为树形表格
        
        // 超级管理员导航项配置
        if (session('auser_id') == 1) {
            $config['navItems'][] = [  
                'url'    => url('rule/lst'), // 权限列表URL
                'title'  => lang('permission_list'), // 权限列表标题
                'active' => true // 激活状态
            ];
        }
        
        return view('common/list', array_merge($config)); // 渲染视图
    }

    // -------------------------------------------------------------------------
    // 表单处理方法
    // -------------------------------------------------------------------------
    public function form(?int $id = null)
    {
        // 非AJAX请求渲染视图
        if (!Request::isAjax()) {
            $model = $id ? RyAuthRuleModel::find($id) : null; // 查找模型
            $currentData = $model ? $model->getData() : []; // 获取数据
            $pageTitle = $id ? lang('edit_permission') : lang('add_permission'); // 页面标题
            $currentParentId = Request::param('addpid/d', 0); // 当前父级ID

            // 字段选项配置
            $fieldsOptions = [
                'pid' => RyAuthRuleModel::getTreeData(), // 父级权限选项
            ];

            $formTabs = RyAuthRuleFields::getFormTabs(); // 获取表单配置
            $this->assignOptionsForFields($formTabs, $fieldsOptions); // 动态赋值选项

            return parent::renderFormView( // 调用基类渲染方法
                $formTabs,
                $currentData,
                $pageTitle,
                $currentParentId
            );
        }

        // AJAX请求处理数据提交
        try {
            $data = Request::param(); // 获取请求数据
            
            // 调用基类处理表单提交
            $result = parent::handleFormSubmit($data, $id);

            $adTitle = $data['title'] ?? ($id ? "ID:{$id}" : lang('new_permission')); // 权限标题
            $actionType = $id ? lang('edit') : lang('add'); // 操作类型
            $this->logFormAction("{$actionType}" . lang('permission') . "《{$adTitle}》"); // 记录日志

            return json([
                'code' => 200,
                'msg' => "{$actionType}" . lang('permission_success'), // 成功消息
                'url' => url('rule/lst')->build(), // 返回URL
                'data' => $result['data'] ?? [] // 返回数据
            ]);
        } catch (Throwable $e) {
            return json([
                'code' => $e->getCode() ?: 500, // 错误代码
                'msg'  => $e->getMessage(), // 错误消息
            ]);
        }
    }

    // -------------------------------------------------------------------------
    // 删除权限规则方法（删除节点及子节点）
    // -------------------------------------------------------------------------
    public function del($ids)
    {
        try {
            Db::startTrans(); // 开始事务

            // 参数处理增强
            $normalizedIds = $this->normalizeIds($ids); // 标准化ID参数

            if (empty($normalizedIds)) {
                return ry_error(lang('invalid_delete_params')); // 无效参数
            }

            // 获取所有子节点ID（处理多ID）
            $allIds = [];
            foreach ($normalizedIds as $rootId) {
                $childIds = $this->getAllChildIds($rootId, 'pid'); // 获取所有子节点ID
                $allIds   = array_merge($allIds, $childIds); // 合并ID
            }
            $allIds = array_unique($allIds); // 去重

            // 执行删除并检查结果
            $result = $this->baseDel($allIds); // 调用基类删除方法

            // 记录操作日志
            $this->logPermissionAction('delete_permission'); // 记录权限操作日志

            Db::commit(); // 提交事务
            return $result; // 返回结果

        } catch (Throwable $e) {
            Db::rollback(); // 回滚事务
            Log::error('删除失败：' . $e->getMessage()); // 记录错误
            return ry_error(lang('delete_failed') . $e->getMessage()); // 返回错误
        }
    }

    // -------------------------------------------------------------------------
    // 私有辅助方法
    // -------------------------------------------------------------------------

    /** 记录权限操作日志 */
    private function logPermissionAction(string $module): void
    {
        RyActionLog::create([
            'uname' => session('auser_name'), // 用户名
            'groupid' => session('auser_gid'), // 用户组ID
            'module' => lang($module), // 操作模块
            'action' => lang('delete'), // 操作类型
            'data_id' => session('auser_id'), // 数据ID
            'description' => lang('permission_operation'), // 操作描述
            'ip' => Request::ip(), // 操作IP
            'os' => $this->getOsInfo(), // 操作系统
        ]);
    }

    /** 获取操作系统信息 */
    private function getOsInfo(): string
    {
        return get_os_info() ?: 'Unknown'; // 获取OS信息
    }
}