<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

namespace app\royaladmin\controller;

use app\royaladmin\common\Baksql;
use app\royaladmin\controller\RyBase;
use think\facade\Request;
use think\facade\Config;
use app\royaladmin\model\RyActionLog;
use Throwable;

// -----------------------------------------------------------------------------
// 数据库备份管理控制器
// 负责数据库备份、恢复、下载和删除操作
// -----------------------------------------------------------------------------
class RyBak extends RyBase
{
    // -------------------------------------------------------------------------
    // 允许的操作类型白名单
    // -------------------------------------------------------------------------
    private const ALLOWED_ACTIONS = ['backup', 'dowonload', 'restore', 'del'];

    // -------------------------------------------------------------------------
    // 数据库备份管理主入口
    // -------------------------------------------------------------------------
    public function bak()
    {
        $action = in_array(input('type'), self::ALLOWED_ACTIONS, true) ? input('type') : ''; // 验证操作类型
        $backup = new Baksql(Config::get('database')); // 初始化备份类

        if ($action) {
            return $this->handleActionRequest($backup, $action); // 处理操作请求
        }

        return view('/ry_bak', [
            'list' => array_reverse($backup->getFileList()) // 获取备份文件列表（倒序）
        ]);
    }

    // -------------------------------------------------------------------------
    // 处理操作请求（带安全验证）
    // -------------------------------------------------------------------------
    private function handleActionRequest(Baksql $backup, string $action)
    {
        try {
            $fileName = $backup->sanitizeFilename(input('name', '')); // 安全过滤文件名
            
            switch ($action) {
                case 'backup':
                    $result = $backup->executeBackup(); // 执行备份操作
                    $this->logBackupAction('backup_management', 'add', lang('backup_added')); // 记录备份日志
                    break;
                case 'dowonload':
                    $result = $backup->downloadBackup($fileName); // 执行下载操作
                    break;
                case 'restore':
                    $result = $backup->restoreBackup($fileName); // 执行恢复操作
                    $this->logBackupAction('backup_management', 'restore', lang('backup_restored')); // 记录恢复日志
                    break;
                case 'del':
                    $result = $backup->deleteBackup($fileName); // 执行删除操作
                    $this->logBackupAction('backup_management', 'delete', lang('backup_deleted', [$result])); // 记录删除日志
                    break;
                default:
                    throw new \RuntimeException(lang('invalid_action_type')); // 无效操作类型
            }

            return ry_success($result); // 返回成功结果
        } catch (Throwable $e) {
            return ry_error($e->getMessage()); // 返回错误信息
        }
    }

    // -------------------------------------------------------------------------
    // 私有辅助方法
    // -------------------------------------------------------------------------

    /** 记录备份操作日志 */
    private function logBackupAction(string $module, string $action, string $description): void
    {
        RyActionLog::create([
            'uname' => session('auser_name'), // 操作用户名
            'groupid' => session('auser_gid'), // 用户组ID
            'module' => lang($module), // 操作模块
            'action' => lang($action), // 操作类型
            'data_id' => session('auser_id'), // 用户ID
            'description' => $description, // 操作描述
            'ip' => Request::ip(), // 操作IP
            'os' => $this->getOsInfo(), // 操作系统信息
        ]);
    }

    /** 获取操作系统信息 */
    private function getOsInfo(): string
    {
        return get_os_info() ?: 'Unknown'; // 获取OS信息
    }
}