<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

namespace app\royaladmin\controller;

use app\royaladmin\controller\RyBase;
use think\facade\Db;
use think\facade\Log;
use think\facade\Request;
use app\royaladmin\model\RyArticle as RyArticleModel;
use app\royaladmin\model\RyCate as RyCateModel;
use app\royaladmin\validate\RyCate as RyCateValidate;
use Throwable;

// -----------------------------------------------------------------------------
// 栏目管理控制器
// 负责栏目的增删改查及树形结构管理
// -----------------------------------------------------------------------------
class RyCate extends RyBase
{
    // -------------------------------------------------------------------------
    // 类常量与属性
    // -------------------------------------------------------------------------
    protected $modelClass    = RyCateModel::class;     // 数据模型类
    protected $validateClass = RyCateValidate::class; // 数据验证类

    // -------------------------------------------------------------------------
    // 获取树形数据
    // -------------------------------------------------------------------------
    public function getTreeData($withPrefix = false)
    {
        // 调用模型方法获取树形分类数据
        return RyCateModel::getTreeData(
            withPrefix: $withPrefix,                    // 是否添加前缀
            fields: ['id,pid,thumb,ename,background,status,title,type,sort,page_tmp,lst_tmp,article_tmp,navmenu'], // 查询字段
        );
    }

    // -------------------------------------------------------------------------
    // 获取栏目数据接口
    // -------------------------------------------------------------------------
    public function getCateData()
    {
        // 获取原始分类树形数据
        $data = $this->getTreeData();
        
        // 收集所有分类ID（包括子分类）
        $cateIds = [];
        $tempStack = $data; // 使用栈结构处理嵌套分类
        while (!empty($tempStack)) {
            $item = array_shift($tempStack);           // 取出栈顶元素
            $cateIds[] = $item['id'];                  // 记录分类ID
            if (!empty($item['children'])) {
                foreach ($item['children'] as $child) {
                    $tempStack[] = $child;             // 子分类入栈
                }
            }
        }
        $cateIds = array_unique($cateIds);             // 去重分类ID
    
        // 批量查询文章统计（仅当有分类时）
        $counts = [];
        if (!empty($cateIds)) {
            $counts = RyArticleModel::where('cateid', 'in', $cateIds)
                ->group('cateid')                      // 按分类分组
                ->column('COUNT(*) as total', 'cateid'); // 获取文章数量
        }
    
        // 递归处理所有分类节点（使用引用修改原数组）
        $processingQueue = [];
        foreach ($data as &$item) {
            $processingQueue[] = &$item;               // 引用入队
        }
        
        while (!empty($processingQueue)) {
            $item = &$processingQueue[0];              // 取队列首元素
            // 设置当前分类文章统计
            $item['art_count'] = $counts[$item['id']] ?? 0;
            
            // 处理子分类
            if (!empty($item['children'])) {
                foreach ($item['children'] as &$child) {
                    $processingQueue[] = &$child;      // 子分类入队
                }
            }
            array_shift($processingQueue);             // 移除已处理元素
        }
    
        return json([
            'code' => 0,
            'msg'  => 'success',
            'data' => $data,
        ]);
    }

    // -------------------------------------------------------------------------
    // 栏目列表页面
    // -------------------------------------------------------------------------
    public function lst()
    {
        // 获取列表配置并渲染视图
        $config = \app\royaladmin\fields\list\RyCateList::getListConfig();
        return view('common/list', array_merge($config));
    }

    // -------------------------------------------------------------------------
    // 表单处理方法
    // -------------------------------------------------------------------------
    public function form(?int $id = null)
    {
        // 非AJAX请求渲染表单视图
        if (!Request::isAjax()) {
            $model = $id ? RyCateModel::find($id) : null;          // 查找模型
            $currentData = $model ? $model->getData() : [];        // 获取数据
            $pageTitle = $id ? lang('edit_category') : lang('add_category'); // 页面标题

            // 字段选项配置
            $fieldsOptions = [
                'pid' => RyCateModel::getTreeData(),               // 父级栏目选项
                'jump_id' => RyCateModel::getTreeData(),           // 跳转栏目选项
            ];

            $formTabs = \app\royaladmin\fields\form\RyCateFields::getFormTabs(); // 获取表单配置
            $this->assignOptionsForFields($formTabs, $fieldsOptions); // 动态赋值选项

            $currentParentId = $id
                ? ($currentData['pid'] ?? 0)                       // 编辑时使用当前父ID
                : Request::param('addpid/d', 0);                   // 新增时使用传入参数

            return parent::renderFormView(                         // 调用基类渲染方法
                $formTabs,
                $currentData,
                $pageTitle,
                $currentParentId
            );
        }

        // AJAX请求处理表单提交
        try {
            $data = Request::param();                              // 获取请求数据
            $model = new RyCateModel();                            // 实例化模型
            $data = $model->processFormData($data, $id);           // 处理表单数据
            $result = parent::handleFormSubmit($data, $id);        // 调用基类提交处理

            $actionType = $id ? lang('edit') : lang('add');        // 操作类型
            $this->logFormAction("{$actionType}" . lang('category') . "《{$data['title']}》"); // 记录日志

            return json($result + [                                // 返回JSON结果
                'require_full_reload' => true,                     // 需要完全刷新
                'url' => url('cate/lst')->build()                  // 跳转URL
            ]);
        } catch (Throwable $e) {
            return json([
                'code' => $e->getCode() ?: 500,                    // 错误代码
                'msg'  => $e->getMessage(),                        // 错误消息
            ]);
        }
    }

    // -------------------------------------------------------------------------
    // 栏目删除方法
    // -------------------------------------------------------------------------
    public function del($ids)
    {
        $ids = (int)$ids;                                          // 转换为整数

        Db::startTrans();                                          // 开始事务
        try {
            if ($ids <= 0) {
                return ry_error(lang('invalid_category_id'));      // 无效ID检查
            }

            $cate = RyCateModel::find($ids);                       // 查找栏目
            if (!$cate) {
                return ry_error(lang('category_not_exists'));      // 栏目不存在
            }

            $allIds = $this->getAllChildIds($ids, 'pid');          // 获取所有子栏目ID
            $allIds[] = $ids;                                      // 包含当前栏目
            $allIds = array_unique($allIds);                       // 去重

            // 检查栏目下是否有数据 - 修复 getBatchRealCount 不存在的问题
            $totalCount = RyArticleModel::where('cateid', 'in', $allIds)->count(); // 直接统计文章数量

            if ($totalCount > 0) {
                $errorMsg = lang('cannot_delete_category_with_data', [$totalCount]); // 错误消息
                return json(['code' => 403, 'msg' => $errorMsg]);  // 返回禁止删除
            }

            RyCateModel::destroy($allIds);                         // 批量删除

            $this->logFormAction(lang('delete_category_log', [$cate->title, count($allIds) - 1])); // 记录日志

            Db::commit();                                          // 提交事务
            return ry_success(lang('delete_category_success', [$cate->title, count($allIds) - 1]), [
                'cate_count' => count($allIds)                     // 返回删除数量
            ]);
        } catch (Throwable $e) {
            Db::rollback();                                        // 回滚事务
            Log::error("栏目删除失败 [ID:{$ids}]: " . $e->getMessage()); // 记录错误日志
            return ry_error(lang('delete_failed') . $e->getMessage()); // 返回错误
        }
    }
}