<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

namespace app\royaladmin\controller;

use app\royaladmin\controller\RyBase;
use think\facade\Db;
use think\facade\Log;
use think\facade\Request;
use think\exception\ValidateException;
use app\royaladmin\model\RyConf as RyConfModel;
use app\royaladmin\validate\RyConf as RyConfValidate;
use app\royaladmin\fields\form\RyConfFields;
use app\royaladmin\fields\list\RyConfList;
use Throwable;

// -----------------------------------------------------------------------------
// 配置管理控制器
// 负责系统配置字段的增删改查及配置值管理
// -----------------------------------------------------------------------------
class RyConf extends RyBase
{
    // -------------------------------------------------------------------------
    // 类常量与属性
    // -------------------------------------------------------------------------
    protected $modelClass    = RyConfModel::class;     // 数据模型类
    protected $validateClass = RyConfValidate::class; // 数据验证类

    // -------------------------------------------------------------------------
    // 表单处理方法
    // -------------------------------------------------------------------------
    public function form(?int $id = null)
    {
        // 非AJAX请求渲染视图
        if (!Request::isAjax()) {
            $model = $id ? RyConfModel::find($id) : null; // 查找模型
            $currentData = $model ? $model->getData() : []; // 获取数据
            $pageTitle = $id ? lang('edit_config_field') : lang('add_config_field'); // 页面标题
            $currentParentId = Request::param('model/d', 0); // 当前模型ID

            $formTabs = RyConfFields::getFormTabs(); // 获取表单配置

            return parent::renderFormView($formTabs, $currentData, $pageTitle, $currentParentId); // 调用基类渲染
        }

        // AJAX请求处理数据提交
        try {
            $data = Request::param(); // 获取请求数据
            
            // 特殊处理：确保字段名唯一性
            if (empty($id) && isset($data['ename'])) {
                $this->validateFieldNameUnique($data['ename']); // 验证字段名唯一
            }

            // 调用基类处理表单提交
            $result = parent::handleFormSubmit($data, $id);
            
            // 记录操作日志
            $fieldTitle = $data['title'] ?? ($id ? "ID:{$id}" : lang('new_field'));
            $actionType = $id ? lang('edit') : lang('add');
            $this->logFormAction("{$actionType}" . lang('config_field') . "《{$fieldTitle}》");

            return json([
                'code' => 200,
                'msg' => "{$actionType}" . lang('config_field_success'), // 成功消息
                'url' => url('conf/lst', ['model' => $data['model'] ?? 0])->build(), // 返回URL
                'data' => $result['data'] ?? [] // 返回数据
            ]);
        } catch (Throwable $e) {
            return json([
                'code' => $e->getCode() ?: 500, // 错误代码
                'msg'  => $e->getMessage() // 错误消息
            ]);
        }
    }

    // -------------------------------------------------------------------------
    // 获取配置数据接口
    // -------------------------------------------------------------------------
    public function getConfData()
    {
        return $this->getCommonData([
            'searchFields' => [
                'title' => 's',  // 标题搜索（字符串匹配）
                'status' => 'd', // 状态搜索（数字匹配）
                'model' => 'd'   // 模型搜索（数字匹配）
            ], 
            'field' => 'id,title,ename,type,model,is_os,status,sort', // 查询字段
            'order' => 'sort asc' // 排序规则
        ]);
    }

    // -------------------------------------------------------------------------
    // 配置字段列表页面
    // -------------------------------------------------------------------------
    public function lst()
    {
        $config = RyConfList::getListConfig(); // 获取列表配置
        
        // 动态添加行操作（根据权限）
        if (session('auser_id') == 1) {
            $config['actions'][] = [
                'url' => url('conf/del') . '?id=', // 删除URL
                'text' => lang('delete'), // 删除文本
                'color' => 'red', // 颜色
                'params' => [], // 参数
                'event' => 'del' // 事件
            ];
        }
        
        return view('common/list', $config); // 渲染视图
    }

    // -------------------------------------------------------------------------
    // 配置管理核心方法
    // -------------------------------------------------------------------------
    public function conf()
    {
        // POST请求处理配置更新
        if (Request::isPost()) {
            return $this->handleConfigUpdate(); // 处理配置更新
        }

        // 参数获取与类型处理
        $is_en = Request::param('is_en/d', null); // 是否启用
        $model = Request::param('model/d', null); // 模型类型

        // 构建基础查询
        $query = RyConfModel::where('status', 1); // 只查询启用的配置

        // 动态添加分类条件
        if (!is_null($is_en)) {
            if (!in_array($is_en, [0, 1])) {
                throw new ValidateException(lang('invalid_category_param')); // 非法分类参数
            }
            $query->where('is_en', $is_en); // 启用状态筛选
        }
        if (!is_null($model)) {
            if (!in_array($model, [1,2,3,4])) {
                throw new ValidateException(lang('invalid_category_param')); // 非法分类参数
            }
            $query->where('model', $model); // 模型类型筛选
        }

        // 获取配置数据
        $configs = $query->order('sort ASC') 
            ->select()
            ->toArray();

        return view('/ry_conf', [
            'Confs' => $configs, // 配置数据
            'model' => $model,   // 模型类型
            'is_en' => $is_en ?? 'all' // 启用状态
        ]);
    }

    // -------------------------------------------------------------------------
    // 删除配置字段方法
    // -------------------------------------------------------------------------
    public function del($ids)
    {
        try {
            Db::startTrans(); // 开始事务

            $normalizedIds = $this->normalizeIds(Request::param('ids')); // 标准化ID参数

            if (empty($normalizedIds)) {
                return ry_error(lang('invalid_delete_params')); // 无效删除参数
            }
            
            // 查询当前数据
            $confs = RyConfModel::where('id', 'in', $normalizedIds)->select();

            // 系统字段保护
            $protectedFields = [];
            foreach ($confs as $conf) {
                if ($conf->is_os == 1) {
                    $protectedFields[] = $conf->ename; // 收集受保护字段
                }
            }
            
            if (!empty($protectedFields)) {
                Db::rollback(); // 回滚事务
                return ry_error(lang('system_field_protected', [implode(',', $protectedFields)])); // 系统字段保护
            }

            // 执行删除
            $deleteCount = RyConfModel::where('id', 'in', $normalizedIds)->delete();
            
            if ($deleteCount === 0) {
                Db::rollback(); // 回滚事务
                return ry_error(lang('no_data_to_delete')); // 没有可删除数据
            }

            // 记录操作日志
            $this->logFormAction(lang('delete_config_fields_log', [$deleteCount]));

            Db::commit(); // 提交事务
            return ry_success(lang('delete_config_fields_success', [$deleteCount])); // 删除成功

        } catch (Throwable $e) {
            Db::rollback(); // 回滚事务
            Log::error('删除失败：' . $e->getMessage()); // 记录错误
            return ry_error(lang('delete_failed') . $e->getMessage()); // 删除失败
        }
    }

    // -------------------------------------------------------------------------
    // 私有辅助方法
    // -------------------------------------------------------------------------

    /** 处理配置更新 */
    private function handleConfigUpdate()
    {
        $data = Request::param(); // 获取请求数据
        $updateData = $this->prepareConfigData($data); // 准备配置数据

        // 批量更新配置
        $updateCount = 0;
        foreach ($updateData as $item) {
            $result = RyConfModel::update(['value' => $item['value']], ['id' => $item['id']]); // 更新配置值
            if ($result) $updateCount++; // 计数
        }

        // 记录操作日志
        $this->logFormAction(lang('update_config_items_log', [$updateCount]));

        return json([
            'code' => 200,
            'msg' => lang('config_update_success'), // 配置更新成功
            'require_full_reload' => true // 需要完全刷新
        ]);
    }

    /** 准备配置数据 */
    private function prepareConfigData(array $formData): array
    {
        $processedData = [];

        try {
            // 获取当前模型下的所有配置项
            $configItems = RyConfModel::where('is_en', $formData['is_en'] ?? 0)
                ->column('id,ename,type,values', 'ename'); // 获取配置项

            foreach ($formData as $field => $rawValue) {
                if (!isset($configItems[$field])) continue; // 跳过不存在的配置项

                $config = $configItems[$field];
                $processedValue = '';

                // 根据不同类型处理字段值
                switch ($config['type']) {
                    case 4: // 复选框
                        $validOptions = explode(',', $config['values']); // 有效选项
                        $values = is_array($rawValue) ? $rawValue : explode(',', $rawValue); // 处理值
                        $filtered = array_intersect($values, $validOptions); // 过滤有效值
                        $processedValue = implode(',', array_map('htmlspecialchars', $filtered)); // 安全处理
                        break;

                    case 3: // 单选按钮
                    case 5: // 下拉菜单
                        $validOptions = explode(',', $config['values']); // 有效选项
                        $cleanValue = htmlspecialchars((string) $rawValue); // 安全处理
                        if (!in_array($cleanValue, $validOptions, true)) {
                            throw new \RuntimeException(lang('invalid_option', [$field])); // 无效选项
                        }
                        $processedValue = $cleanValue; // 设置处理后的值
                        break;

                    default: // 文本类型
                        $processedValue = htmlspecialchars((string) $rawValue); // 安全处理
                }

                $processedData[] = [
                    'id' => $config['id'], // 配置ID
                    'value' => $processedValue, // 处理后的值
                ];
            }
        } catch (Throwable $e) {
            Log::error("配置处理失败: {$e->getMessage()}"); // 记录错误
            throw new \RuntimeException(lang('config_data_process_exception')); // 配置数据处理异常
        }

        return $processedData;
    }

    /** 验证字段名唯一性 */
    private function validateFieldNameUnique(string $fieldName): void
    {
        $exists = RyConfModel::where('ename', $fieldName)->count(); // 检查字段名是否存在
        if ($exists > 0) {
            throw new ValidateException(lang('field_name_exists', [$fieldName])); // 字段名已存在
        }
    }
}