<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

namespace app\royaladmin\controller;

use app\royaladmin\controller\RyBase;
use app\royaladmin\model\RyArticle as RyArticleModel;
use app\royaladmin\model\RyCate as RyCateModel;
use app\royaladmin\model\RyAd as RyAdModel;
use think\facade\View;
use Throwable;

// -----------------------------------------------------------------------------
// 后台首页控制器
// 负责仪表盘数据展示、系统信息和统计功能
// -----------------------------------------------------------------------------
class RyIndex extends RyBase
{
    // -------------------------------------------------------------------------
    // 后台首页方法
    // -------------------------------------------------------------------------
    public function index()
    {
        // 获取最新文章（15条）
        $latestArticles = RyArticleModel::order('id', 'desc')
            ->limit(15)
            ->select();

        // 仪表盘卡片配置
        $cardsConfig = $this->getCardsConfig();

        // 快捷方式配置
        $shortcuts = $this->getShortcuts();

        // 系统安全规则配置
        $securityRules = $this->getSecurityRules();

        // 动态生成最新动态信息
        $latestNews = $this->generateLatestNews();

        // 计算网站运行时间
        $uptime = $this->calculateUptime();

        // 计算网站到期时间
        $expireDate = $this->calculateExpireDate();

        // 获取系统环境信息
        $systemInfo = $this->getSystemInfo($uptime);

        // 页面标题和模块标题
        $pageTitles = [
            'page_title' => lang('admin_home'),
            'website_data_overview' => lang('website_data_overview'),
            'quick_operations' => lang('quick_operations'),
            'system_information' => lang('system_information'),
            'website_expiry_countdown' => lang('website_expiry_countdown'),
            'scan_qrcode_to_add_wechat' => lang('scan_qrcode_to_add_wechat'),
            'notes' => lang('notes'),
            'technical_support' => lang('technical_support'),
            'support_us' => lang('support_us')
        ];

        return View::fetch('/ry_index', [
            // 核心统计模块数据
            'cardsConfig'   => $cardsConfig,
            'shortcuts'     => $shortcuts,
            'securityRules' => $securityRules,

            // 最新数据和动态
            'latestArticles' => $latestArticles,
            'latestNews'    => $latestNews,

            // 网站到期时间
            'expire_date'   => $expireDate,

            // 系统环境信息
            'system_info'   => $systemInfo,

            // 页面标题
            'pageTitles'    => $pageTitles,

            // JavaScript多语言文本
            'jsLang' => [
                'day' => lang('day'),
                'hour' => lang('hour'),
                'minute' => lang('minute'),
                'second' => lang('second'),
                'website_expiry' => lang('website_expiry'),
                'thank_you_for_support' => lang('thank_you_for_support'),
                'donation_help' => lang('donation_help'),
                'wechat_consultation' => lang('wechat_consultation'),
                'scan_qrcode_to_add_wechat' => lang('scan_qrcode_to_add_wechat'),
                'wechat_qrcode' => lang('wechat_qrcode'),
                'qq_consultation' => lang('qq_consultation'),
                'qq_number' => lang('qq_number'),
                'consult_immediately' => lang('consult_immediately'),
                'system_version' => lang('system_version'),
                'php_version' => lang('php_version'),
                'server_software' => lang('server_software'),
                'database' => lang('database'),
                'login_ip' => lang('login_ip'),
                'server_time' => lang('server_time'),
                'uptime' => lang('uptime')
            ]
        ]);
    }

    // -------------------------------------------------------------------------
    // 获取仪表盘卡片配置
    // -------------------------------------------------------------------------
    private function getCardsConfig(): array
    {
        return [
            'article_pending' => [
                'title' => lang('pending_articles'),
                'count' => RyArticleModel::where('status', 0)->count(),
                'url'   => url('article/lst', ['status' => 0]),
                'icon'  => 'layui-icon-date',
                'badge' => ['color' => 'red', 'text' => lang('review')],
                'gradient' => ['start' => '#FF5722', 'end' => '#FF8A65']
            ],
            'article_total' => [
                'title' => lang('all_articles'),
                'count' => RyArticleModel::count(),
                'url'   => url('article/lst'),
                'icon'  => 'layui-icon-tabs',
                'badge' => ['color' => 'orange', 'text' => lang('total')],
                'gradient' => ['start' => '#FFB800', 'end' => '#FFD54F']
            ],
            'category_total' => [
                'title' => lang('all_categories'),
                'count' => RyCateModel::count(),
                'url'   => url('cate/lst'),
                'icon'  => 'layui-icon-component',
                'badge' => ['color' => 'blue', 'text' => lang('total')],
                'gradient' => ['start' => '#1E9FFF', 'end' => '#4FC3F7']
            ],
            'advertisement_total' => [
                'title' => lang('image_ads'),
                'count' => RyAdModel::count(),
                'url'   => url('ad/lst'),
                'icon'  => 'layui-icon-picture',
                'badge' => ['color' => 'green', 'text' => lang('total')],
                'gradient' => ['start' => '#16baaa', 'end' => '#4DB6AC']
            ]
        ];
    }
    // -------------------------------------------------------------------------
    // 获取快捷方式配置
    // -------------------------------------------------------------------------
    private function getShortcuts(): array
    {
        return [
            [
                'title' => lang('category_list'),
                'url' => url('cate/lst'),
                'icon' => 'layui-icon-component',
                'class' => 'royal-normal-link'
            ],
            [
                'title' => lang('article_list'),
                'url' => url('article/lst'),
                'icon' => 'layui-icon-tabs',
                'class' => 'royal-normal-link'
            ],
            [
                'title' => lang('website_settings'),
                'url' => url('conf/conf', ['model' => 1]),
                'icon' => 'layui-icon-set',
                'class' => 'royal-normal-link'
            ],
            [
                'title' => lang('image_ads'),
                'url' => url('ad/lst'),
                'icon' => 'layui-icon-picture',
                'class' => 'royal-normal-link'
            ],
            [
                'title' => lang('user_management'),
                'url' => url('auser/lst'),
                'icon' => 'layui-icon-user',
                'class' => 'royal-normal-link'
            ],
            [
                'title' => lang('friend_links'),
                'url' => url('link/lst'),
                'icon' => 'layui-icon-link',
                'class' => 'royal-normal-link'
            ],
            [
                'title' => lang('clear_cache'),
                'url' => 'javascript:void(0);',
                'icon' => 'layui-icon-fonts-clear',
                'class' => 'royal-layer-msg layui-text-warm',
                'data-url' => url('common/clearCache'),
            ],
            [
                'title' => lang('backup_data'),
                'url' => 'javascript:void(0);',
                'icon' => 'layui-icon-upload-drag',
                'class' => 'royal-layer-msg layui-text-blue',
                'data-url' => url('bak/bak', ['type' => 'backup']),
            ],
        ];
    }
    // -------------------------------------------------------------------------
    // 获取系统安全规则配置
    // -------------------------------------------------------------------------
    private function getSecurityRules(): array
    {
        return [
            [
                'title'   => lang('data_protection'),
                'content' => lang('data_protection_content')
            ],
            [
                'title'   => lang('operation_specifications'),
                'content' => lang('operation_specifications_content')
            ],
            [
                'title'   => lang('login_environment'),
                'content' => lang('login_environment_content')
            ],
            [
                'title'   => lang('account_protection'),
                'content' => lang('account_protection_content')
            ],
            [
                'title'   => lang('password_management'),
                'content' => lang('password_management_content')
            ]
        ];
    }

    // -------------------------------------------------------------------------
    // 生成最新动态信息
    // -------------------------------------------------------------------------
    private function generateLatestNews(): array
    {
        $news = [];

        // 获取最新文章
        $latestArticles = RyArticleModel::order('id', 'desc')
            ->limit(10)
            ->select();

        foreach ($latestArticles as $article) {
            $news[] = [
                'time' => date('Y-m-d H:i:s', strtotime($article->create_time)),
                'content' => lang('published_new_article', [$article->title]),
                'icon' => 'layui-icon-tabs'
            ];
        }

        // 获取最新栏目
        $latestCategories = RyCateModel::order('id', 'desc')
            ->limit(10)
            ->select();

        foreach ($latestCategories as $category) {
            $news[] = [
                'time' => date('Y-m-d H:i:s', strtotime($category->create_time)),
                'content' => lang('added_new_category', [$category->title]),
                'icon' => 'layui-icon-component'
            ];
        }

        // 按时间倒序排序
        usort($news, function ($a, $b) {
            return strtotime($b['time']) - strtotime($a['time']);
        });

        // 只返回最新的10条
        return array_slice($news, 0, 10);
    }

    // -------------------------------------------------------------------------
    // 计算网站运行时间
    // -------------------------------------------------------------------------
    private function calculateUptime(): string
    {
        // 假设网站安装时间为2024年1月1日
        $installTime = strtotime('2024-01-01 00:00:00');
        $currentTime = time();
        $diff = $currentTime - $installTime;

        $days = floor($diff / (60 * 60 * 24));
        $hours = floor(($diff % (60 * 60 * 24)) / (60 * 60));
        $minutes = floor(($diff % (60 * 60)) / 60);

        return lang('uptime_format', [$days, $hours, $minutes]);
    }

    // -------------------------------------------------------------------------
    // 计算网站到期时间
    // -------------------------------------------------------------------------
    private function calculateExpireDate(): string
    {
        // 假设网站到期时间为一年后
        $expireTime = strtotime('+1 year');
        return date('Y-m-d', $expireTime);
    }

    // -------------------------------------------------------------------------
    // 获取系统环境信息
    // -------------------------------------------------------------------------
    private function getSystemInfo(string $uptime): array
    {
        return [
            'user_agent'       => $_SERVER['HTTP_USER_AGENT'] ?? lang('unknown'),
            'php_version'      => 'PHP-' . PHP_VERSION,
            'server_software'  => $_SERVER['SERVER_SOFTWARE'] ?? lang('unknown'),
            'database'         => 'MySQL 5.7+',
            'remote_ip'        => $_SERVER['REMOTE_ADDR'] ?? lang('unknown'),
            'server_time'      => date('Y-m-d H:i:s'),
            'uptime'           => $uptime,
            'thinkphp_version' => app()->version(),
            'os'               => PHP_OS,
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'memory_limit'     => ini_get('memory_limit')
        ];
    }

    // -------------------------------------------------------------------------
    // 获取系统统计信息
    // -------------------------------------------------------------------------
    public function getStats()
    {
        try {
            $stats = [
                'article' => [
                    'total' => RyArticleModel::count(),
                    'pending' => RyArticleModel::where('status', 0)->count(),
                    'published' => RyArticleModel::where('status', 1)->count()
                ],
                'category' => [
                    'total' => RyCateModel::count()
                ],
                'ad' => [
                    'total' => RyAdModel::count()
                ]
            ];

            return json([
                'code' => 1,
                'msg' => lang('get_success'),
                'data' => $stats
            ]);
        } catch (Throwable $e) {
            return json([
                'code' => 0,
                'msg' => lang('get_failed') . $e->getMessage()
            ]);
        }
    }
}
