<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);
namespace app\royaladmin\controller;

use app\royaladmin\model\RyAuser;
use think\facade\Session;
use think\facade\View;
use think\facade\Request;

class RyLogin
{
    // -------------------------------------------------------------------------
    // 常量配置
    // -------------------------------------------------------------------------
    private const SESSION_KEY = 'auser_id';
    private const HOME_URL = '/royalcms/website/home';
    private const LOGIN_URL = '/royalcms/website/login/index';

    // -------------------------------------------------------------------------
    // 主入口方法
    // -------------------------------------------------------------------------
    public function index()
    {
        return Session::has(self::SESSION_KEY) 
            ? $this->redirectToHome() 
            : View::fetch('/ry_login');
    }

    // -------------------------------------------------------------------------
    // 登录验证处理
    // -------------------------------------------------------------------------
    public function loginCheck()
    {
        if (!Request::isAjax()) {
            return $this->jsonResponse(400, lang('illegal_request'));
        }

        $data = Request::param();

        // 验证码检查
        if (!$this->validateCaptcha($data['code'] ?? '')) {
            return $this->jsonResponse(400, lang('captcha_error'));
        }

        // 用户验证
        $status = (new RyAuser())->validateUser($data);
        return $this->handleLoginStatus($status);
    }

    // -------------------------------------------------------------------------
    // 用户退出处理
    // -------------------------------------------------------------------------
    public function loginOut()
    {
        Session::clear();
        return $this->jsonResponse(200, lang('logout_success'), ['url' => self::LOGIN_URL]);
    }

    // -------------------------------------------------------------------------
    // 私有辅助方法
    // -------------------------------------------------------------------------

    /** 重定向到首页 */
    private function redirectToHome()
    {
        return ry_alert(lang('already_logged_in'), self::HOME_URL);
    }

    /** JSON响应封装 */
    private function jsonResponse(int $code, string $msg, array $data = [])
    {
        return json(compact('code', 'msg', 'data'));
    }

    /** 验证码验证 */
    private function validateCaptcha(?string $code): bool
    {
        return !empty($code) && captcha_check($code);
    }

    /** 处理登录状态 */
    private function handleLoginStatus(int $status)
    {
        return match($status) {
            1 => $this->jsonResponse(200, lang('login_success'), ['url' => self::HOME_URL]),
            3 => $this->jsonResponse(400, lang('account_not_exists')),
            4 => $this->jsonResponse(400, lang('account_disabled')),
            default => $this->jsonResponse(400, lang('login_failed'))
        };
    }
}