<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------
namespace app\royaladmin\controller;
 
use app\royaladmin\controller\RyBase;
use app\royaladmin\validate\RyUpload as RyUploadValidate;
use think\facade\Db; 
use think\facade\Log; 
use think\facade\Request; 
use think\facade\Filesystem;
use think\Image;
use think\exception\FileException;

class RyUpload extends RyBase
{
    // -------------------------------------------------------------------------
    // 上传配置常量定义
    // -------------------------------------------------------------------------
    const MAX_IMAGE_SIZE = 5 * 1024 * 1024;      // 5MB
    const MAX_FILE_SIZE  = 50 * 1024 * 1024;     // 50MB
    const ALLOW_IMAGE_EXT = 'jpg,png,jpeg,gif';  // 允许的图片格式
    const ALLOW_FILE_EXT  = 'zip,rar,7z,pdf,doc,docx,xls,xlsx,exe,mp3,mp4,m4v,jpg,png,jpeg,gif'; // 允许的文件格式

    // -------------------------------------------------------------------------
    // 统一图片上传方法 
    // -------------------------------------------------------------------------
    public function UploadImg()
    {
        // 接收文件对象
        $file = Request::file('file');

        // 使用统一验证器
        $validate = new RyUploadValidate();
        if (! $validate->scene('image')->check(['image' => $file])) {
            return ry_error($validate->getError());
        }

        try {
            // 批量获取配置参数（减少数据库查询）
            $configs = Db::name('conf')
                ->whereIn('ename', ['water', 'waterimg'])
                ->column('value', 'ename');

            // 上传到本地服务器
            $path     = Filesystem::putFile('images', $file);
            $fullPath = 'uploads/' . $path;

            // 水印处理
            $this->processWatermark($fullPath, $configs);

            // 返回标准化路径
            return json([
                'code' => 200,
                'msg'  => lang('upload_success'),
                'path' => '/' . str_replace('\\', '/', $fullPath),
            ]);

        } catch (\Exception $e) {
            Log::error('图片上传失败：' . $e->getMessage());
            return json([
                'code' => 500, 
                'msg'  => lang('upload_server_error')
            ]);
        }
    }

    // -------------------------------------------------------------------------
    // 删除图片方法
    // -------------------------------------------------------------------------
    public function DeleteImg()
    {
        try {
            $path = Request::post('path');
            
            // 参数验证
            if (empty($path) || !$this->validatePath($path)) {
                return json([
                    'code' => 400, 
                    'msg'  => lang('invalid_parameters')
                ]);
            }

            // 构建完整路径
            $fullPath = $this->getFullPath($path);
            
            // 安全检查
            if (!$this->isSafeUploadPath($fullPath)) {
                return json([
                    'code' => 403, 
                    'msg'  => lang('operation_forbidden')
                ]);
            }

            // 文件存在性检查
            if (!file_exists($fullPath)) {
                return json([
                    'code' => 404, 
                    'msg'  => lang('file_not_exists')
                ]);
            }

            // 执行删除
            if (@unlink($fullPath)) {
                return json([
                    'code' => 200, 
                    'msg'  => lang('delete_success')
                ]);
            }
            
            return json([
                'code' => 500, 
                'msg'  => lang('delete_failed')
            ]);

        } catch (\Exception $e) {
            Log::error('图片删除失败: ' . $e->getMessage());
            return json([
                'code' => 500, 
                'msg'  => lang('server_error')
            ]);
        }
    }

    // -------------------------------------------------------------------------
    // 文件上传方法
    // -------------------------------------------------------------------------
    public function uploadFile()
    {
        try {
            // 文件存在性检查
            $file = Request::file('file');
            if (!$file) {
                return $this->getUploadLimitError();
            }

            // 文件大小验证
            if (!$this->validateFileSize($file->getSize(), self::MAX_FILE_SIZE)) {
                return json([
                    'code' => 500, 
                    'msg'  => lang('file_size_exceed') . '50MB'
                ]);
            }

            // 文件类型安全验证
            if (!$this->validateFileExtension($file, self::ALLOW_FILE_EXT)) {
                return json([
                    'code' => 500, 
                    'msg'  => lang('invalid_file_type')
                ]);
            }

            // 安全存储
            $savename = Filesystem::disk('public')
                ->putFile('uplodfile', $file, 'md5');
            $path = '/uploads/' . str_replace('\\', '/', $savename);

            return json([
                'code' => 200, 
                'msg'  => lang('upload_success'), 
                'path' => $path
            ]);

        } catch (FileException $e) {
            return json([
                'code' => 500, 
                'msg'  => $e->getMessage()
            ]);
        } catch (\Exception $e) {
            return json([
                'code' => 500, 
                'msg'  => lang('upload_failed')
            ]);
        }
    }

    // -------------------------------------------------------------------------
    // 商品图片上传方法
    // -------------------------------------------------------------------------
    public function uploadImages() 
    {
        $file = Request::file('file');
        
        try {
            // 验证规则
            $this->validateImageFile($file);

            // 存储路径
            $saveName = Filesystem::disk('public')->putFile('images/goods', $file);
            $path = '/uploads/' . str_replace('\\', '/', $saveName);
            
            // 返回格式
            return json([
                'code' => 0,
                'data' => [
                    'url'  => Request::domain() . $path,
                    'path' => $path
                ]
            ]);
            
        } catch (\Exception $e) {
            return ry_error(lang('invalid_operation'));
        }
    }

    // -------------------------------------------------------------------------
    // 私有辅助方法
    // -------------------------------------------------------------------------

    /** 处理图片水印 */
    private function processWatermark($imagePath, $configs)
    {
        if ($configs['water'] === '开启' && !empty($configs['waterimg'])) {
            $watermark = app()->getRootPath() . 'public/' . ltrim($configs['waterimg'], '/');
            
            if (file_exists($watermark)) {
                Image::open($imagePath)
                    ->water($watermark, Image::WATER_SOUTHEAST, 80)
                    ->save($imagePath);
            }
        }
    }

    /** 验证文件路径安全性 */
    private function validatePath($path)
    {
        return !empty($path) && is_string($path) && 
               strpos($path, '..') === false && 
               strlen($path) < 500;
    }

    /** 获取完整文件路径 */
    private function getFullPath($path)
    {
        return app()->getRootPath() . 'public/' . ltrim($path, '/');
    }

    /** 检查是否为安全的上传路径 */
    private function isSafeUploadPath($fullPath)
    {
        $uploadDirs = ['uploads/images/', 'uploads/uplodfile/', 'uploads/images/goods/'];
        
        foreach ($uploadDirs as $dir) {
            if (strpos($fullPath, $dir) !== false) {
                return true;
            }
        }
        return false;
    }

    /** 获取上传限制错误信息 */
    private function getUploadLimitError()
    {
        $maxSize = min(
            $this->convertToBytes(ini_get('upload_max_filesize')),
            $this->convertToBytes(ini_get('post_max_size'))
        );
        $maxSizeReadable = round($maxSize / (1024 * 1024), 2) . 'MB';
        
        return json([
            'code' => 500, 
            'msg'  => lang('upload_size_exceed') . $maxSizeReadable . lang('contact_technical')
        ]);
    }

    /** 转换大小单位为字节 */
    private function convertToBytes($size)
    {
        $unit = preg_replace('/[^bkmgtpezy]/i', '', $size);
        $size = preg_replace('/[^0-9\.]/', '', $size);
        
        if ($unit) {
            return round($size * pow(1024, stripos('bkmgtpezy', $unit[0])));
        }
        
        return round($size);
    }

    /** 验证文件大小 */
    private function validateFileSize($fileSize, $maxSize)
    {
        return $fileSize <= $maxSize;
    }

    /** 验证文件扩展名 */
    private function validateFileExtension($file, $allowedExtensions)
    {
        $extension = strtolower($file->extension());
        $allowed = explode(',', $allowedExtensions);
        
        return in_array($extension, $allowed);
    }

    /** 验证图片文件 */
    private function validateImageFile($file)
    {
        validate(['file' => [
            'fileSize' => self::MAX_IMAGE_SIZE,
            'fileExt'  => self::ALLOW_IMAGE_EXT
        ]])->check(['file' => $file]);
    }
}