<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

namespace app\royaladmin\model;

use think\Model;
use think\facade\Session;
use think\facade\Request;
use app\royaladmin\model\RyLog as RyLogModel;

use app\royaladmin\model\traits\TreeBuilder;

class RyAuser extends Model
{
    use TreeBuilder;

    //—————————————————————————————————————————————————
    // 模型配置
    //————————————————————————————————————————————————— 

    protected $name = 'auser';

    // 密码加密盐值常量
    private const PASSWORD_SALT = 'royal';

    //—————————————————————————————————————————————————
    // 获取器
    //—————————————————————————————————————————————————
    public function getLoginTimeAttr($value): string
    {
        return $value ? date('Y-m-d H:i:s', $value) : '';
    }

    //—————————————————————————————————————————————————
    // 模型关联
    //————————————————————————————————————————————————— 
    public function bltAuthGroup()
    {
        return $this->belongsTo(RyAuthGroup::class, 'groupid')
            ->bind(['group_title' => 'title']);
    }


    //—————————————————————————————————————————————————
    // 核心业务方法
    //—————————————————————————————————————————————————
    public function validateUser(array $data): int
    {
        $admin = $this->getAdminByUsername($data['admin_name'] ?? '');

        if (!$admin) return 3;
        if (!$this->validatePassword($admin, $data['admin_password'] ?? '')) return 2;
        if ($admin->status == 0) return 4;

        $this->handleSuccessfulLogin($admin);
        return 1;
    }

    //—————————————————————————————————————————————————
    // 私有辅助方法
    //—————————————————————————————————————————————————

    // 根据用户名获取管理员信息
    private function getAdminByUsername(string $username): ?self
    {
        return $this->where('uname', $username)->find();
    }

    // 验证密码有效性
    private function validatePassword(self $admin, string $password): bool
    {
        $encrypted = sha1(sha1(self::PASSWORD_SALT . $password));
        return $admin->password === $encrypted;
    }

    // 处理登录成功后的系列操作
    private function handleSuccessfulLogin(self $admin): void
    {
        $this->setSessionData($admin);
        $this->updateLoginInfo($admin);
        $this->createLoginLog($admin);
    }

    // 设置会话数据
    private function setSessionData(self $admin): void
    {
        Session::set('auser_name', $admin->uname);
        Session::set('auser_id', $admin->id);
        Session::set('auser_gid', $admin->groupid);
    }

    // 更新最后登录信息
    private function updateLoginInfo(self $admin): void
    {
        $admin->save([
            'login_time' => time(), //登陆时间
            'session_id' => Session::getId(), //session_id 
            'last_active_time' => time(), //最后活动时间
            'login_ip'   => Request::ip()
        ]);
    }

    // 创建登录日志记录
    private function createLoginLog(self $admin): void
    {
        RyLogModel::create([
            'adminid'    => $admin->id,
            'groupid'    => $admin->groupid,
            'uname'      => $admin->uname,
            'login_time' => time(),
            'login_ip'   => Request::ip(),
            'login_os'   => Request::server('HTTP_USER_AGENT')
        ]);
    }
}
