<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 * 
 * @name      RoyalCMS 若伊智能网站构建系统
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 * 
 * 开源授权说明：
 * 允许：个人/商业免费使用、修改、分发、二次开发
 * 允许：基于本系统进行商业项目开发
 *
 * 严禁：直接打包本系统代码进行售卖
 * 严禁：将本系统作为付费产品的一部分分发
 * 严禁：去除版权信息后声称自己是原作者
 * 
 * 法律声明：
 * 任何违反上述规定的行为均构成侵权，我们将采取法律手段维护权益
 * 包括但不限于民事诉讼、刑事举报等法律途径
 * 
 * 请尊重开源精神，共建良好开源环境！
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);
namespace app\royaladmin\model;

use think\Model;
use think\facade\Session;
use think\Collection;
use app\royaladmin\model\traits\TreeBuilder;
class RyAuthGroup extends Model
{
    // +----------------------------------------------------------------------
    // | 引入数结构
    // +----------------------------------------------------------------------
    use TreeBuilder;
    // +----------------------------------------------------------------------
    // | 定义数据表
    // +----------------------------------------------------------------------

    protected $name = 'auth_group';

    // +----------------------------------------------------------------------
    // | 模型配置
    // +----------------------------------------------------------------------

    protected $type = [
        'rules' => 'comma', // 定义自动类型转换
    ];

    // +----------------------------------------------------------------------
    // | 作用域
    // +----------------------------------------------------------------------
    
    // 过滤超级管理员可见性
    public function scopeFilterSuperAdmin($query)
    {
        $gid = Session::get('auser_id', 0);
        if ($gid != 1) {
            $query->where('id', '<>', 1);
        }
        return $query;
    }

    // +----------------------------------------------------------------------
    // | 权限树操作
    // +----------------------------------------------------------------------
    
    // 获取完整权限树（带缓存）
    public static function getPermissionTree(): array
    {
        static $cache = null;
        if ($cache === null) {
            $rules = RyAuthRule::scope('filterStatus')
                ->order('sort , id ASC')
                ->select()
                ->toArray();
            $cache = self::buildTree($rules);
        }
        return $cache;
    }

    // 构建树形结构
    private static function buildTree(array $items, int $pid = 0): array
    {
        $tree = [];
        foreach ($items as $item) {
            if ($item['pid'] == $pid) {
                $children = self::buildTree($items, $item['id']);
                $item['children'] = $children ?: [];
                $tree[] = $item;
            }
        }
        return $tree;
    }

    // +----------------------------------------------------------------------
    // | 权限操作
    // +----------------------------------------------------------------------
    
    // 获取实际有效的权限ID集合
    public function getValidRulesAttribute(): array
    {
        $rules = $this->rules ? explode(',', $this->rules) : [];
        return array_unique(array_filter(array_map('intval', $rules)));
    }

    // 获取完整权限链（包含子权限）
    public function getFullRuleChain(): array
    {
        $selectedRules = $this->valid_rules;
        if (empty($selectedRules)) return [];

        $ruleIds = $selectedRules;
        $childIds = RyAuthRule::whereIn('pid', $selectedRules)
            ->column('id');

        // 递归获取子权限（最多3层）
        for ($i = 0; $i < 6; $i++) {
            if (empty($childIds)) break;
            $ruleIds = array_merge($ruleIds, $childIds);
            $childIds = RyAuthRule::whereIn('pid', $childIds)
                ->column('id');
        }

        return array_unique($ruleIds);
    }

    // +----------------------------------------------------------------------
    // | 树形结构操作
    // +----------------------------------------------------------------------
    
    // 获取所有子组ID（包含自身）
    public static function getAllChildIdsWithSelf($id): array
    {
        $groups = self::select();
        return array_unique(array_merge(
            [intval($id)],
            self::findChildIds($groups, $id)
        ));
    }

    // 递归查找子ID
    private static function findChildIds(Collection $groups, int $pid): array
    {
        $ids = [];
        foreach ($groups as $group) {
            if ($group->pid == $pid) {
                $ids[] = $group->id;
                $ids = array_merge(
                    $ids, 
                    self::findChildIds($groups, $group->id)
                );
            }
        }
        return $ids;
    }

    // +----------------------------------------------------------------------
    // | 权限更新
    // +----------------------------------------------------------------------
    
    // 安全更新权限
    public function safeUpdateRules(array $ruleIds): bool
    {
        try {
            // 清理无效权限ID
            $validIds = RyAuthRule::whereIn('id', $ruleIds)
                ->column('id');

            // 更新规则字段
            $this->rules = implode(',', $validIds);
            return $this->save();
        } catch (\Throwable $e) {
            throw new \RuntimeException('权限更新失败：'.$e->getMessage());
        }
    }

    // +----------------------------------------------------------------------
    // | 类型转换
    // +----------------------------------------------------------------------
    
    // 自定义逗号分隔转换
    public function getCommaAttr($value, $data)
    {
        $field = $data['rules'] ?? '';
        return is_string($field) ? $field : '';
    }

    public function setCommaAttr($value)
    {
        return is_array($value) ? implode(',', $value) : $value;
    }
}