<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

namespace app\royaladmin\model;

use think\Model;

class RyBase extends Model
{
   
    // +----------------------------------------------------------------------
    // | 获取器
    // +----------------------------------------------------------------------

    // 时间获取器
    public function getCreateTimeAttr($value)
    {
        return date('Y-m-d', $value);
    }

    // +----------------------------------------------------------------------
    // | 核心数据处理方法
    // +----------------------------------------------------------------------

    // 处理表单数据
    public function processFormData(array $data, ?int $id = null): array
    {
        $processed = $this->processBasicFields($data);
        $this->autoSetStatus($processed, $id);
        $this->autoGenerateDescription($processed);
        $processed['catesub'] = $this->processCatesubField($processed['catesub'] ?? []);

        return $processed;
    }

    // +----------------------------------------------------------------------
    // | 字段处理方法
    // +----------------------------------------------------------------------

    // 处理基础字段
    protected function processBasicFields(array $data): array
    {
        $processed = [];

        foreach ($data as $key => $value) {
            $processed[$key] = match ($key) {
                'catesub'    => array_unique(array_filter((array)$value, 'is_numeric')),
                'cateid'     => (int)$value,
                'status'     => (int)($value ?? 1),
                'sort'       => (int)($value ?? 0),
                'click'      => $value ?? mt_rand(1000, 9999),
                'create_time' => $this->parseCreateTime($value ?? ''),
                default      => is_string($value) ? trim($value) : $value
            };
        }

        return $processed;
    }

    // 处理创建时间
    protected function parseCreateTime($input): int
    {
        return empty($input) ? time() : (is_numeric($input) ? (int)$input : strtotime($input));
    }

    // 处理副栏目字段
    protected function processCatesubField(array $catesub): string
    {
        return !empty($catesub) ? implode(',', $catesub) : '';
    }

    // +----------------------------------------------------------------------
    // | 自动处理逻辑
    // +----------------------------------------------------------------------

    // 自动设置状态
    protected function autoSetStatus(array &$data, ?int $id = null): void
    {
        if (session('auser_gid') < 3 && is_null($id)) {
            $data['status'] = 1;
        }
    }

    // 自动生成描述
    protected function autoGenerateDescription(array &$data): void
    {
        $descKey = $this->getDescriptionField();
        $contentKey = $this->getContentField();

        if (empty($data[$descKey]) && !empty($data[$contentKey])) {
            $data[$descKey] = $this->generateDescription($data[$contentKey]);
        }
    }

    // 生成描述
    protected function generateDescription(string $content): string
    {
        $cleanContent = strip_tags(str_replace(["\r", "\n"], ' ', $content));
        $length = $this->getDescriptionLength();

        return mb_substr($cleanContent, 0, $length, 'UTF-8') .
            (mb_strlen($cleanContent) > $length ? '...' : '');
    }

    // +----------------------------------------------------------------------
    // | 可配置方法（子类可覆盖）
    // +----------------------------------------------------------------------

    // 获取描述字段名
    protected function getDescriptionField(): string
    {
        return 'description';
    }

    // 获取内容字段名
    protected function getContentField(): string
    {
        return 'content';
    }

    // 获取描述生成长度
    protected function getDescriptionLength(): int
    {
        return 180;
    }

    // +----------------------------------------------------------------------
    // | 搜索器
    // +----------------------------------------------------------------------

    // 栏目ID搜索器
    public function searchCateidAttr($query, $value): void
    {
        if ($value > 0) {
            $query->where('cateid', $value);
        }
    }

    // 状态搜索器
    public function searchStatusAttr($query, $value): void
    {
        $query->where('status', $value);
    }

    // +----------------------------------------------------------------------
    // | 工具方法
    // +----------------------------------------------------------------------

    // 标准化ID数组
    protected function normalizeIds($ids): array
    {
        if (is_array($ids)) {
            return array_filter(array_map('intval', $ids));
        }

        if (is_numeric($ids)) {
            return [(int)$ids];
        }

        if (is_string($ids) && str_contains($ids, ',')) {
            return array_filter(array_map('intval', explode(',', $ids)));
        }

        return [];
    }
}
