<?php
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

namespace app\royaladmin\service;

use think\facade\Db;
use think\facade\Log;
use app\royaladmin\model\RyArticle as RyArticleModel;

class RyArticleService
{
    // -------------------------------------------------------------------------
    // 准备表单数据
    // -------------------------------------------------------------------------

    public function prepareFormData(?RyArticleModel $model): array
    {
        if (!$model) {
            return [];
        }

        $data = $model->getData();
        
        // 处理多选字段
        $data = $this->processMultiSelectFields($data);
        
        // 处理时间字段
        if ($model->create_time) {
            $data['create_time'] = $model->create_time;
        }

        return $data;
    }

    // -------------------------------------------------------------------------
    // 处理多选字段
    // -------------------------------------------------------------------------

    private function processMultiSelectFields(array $data): array
    {
        $multiSelectFields = ['images', 'catesub'];
        
        foreach ($multiSelectFields as $field) {
            if (isset($data[$field]) && is_string($data[$field]) && !empty($data[$field])) {
                $data[$field] = explode(',', $data[$field]);
            } else {
                $data[$field] = [];
            }
        }

        return $data;
    }

    // -------------------------------------------------------------------------
    // 处理表单数据
    // -------------------------------------------------------------------------

    public function processFormData(array $data, ?int $id = null): array
    {
        // 使用模型基类的 processFormData 方法处理数据
        $model = new RyArticleModel();
        $processed = $model->processFormData($data, $id);
        
        // 服务层特定的数据处理
        $processed = $this->sanitizeFormData($processed);
        
        return $processed;
    }

    // -------------------------------------------------------------------------
    // 数据清理和验证
    // -------------------------------------------------------------------------

    private function sanitizeFormData(array $data): array
    {
        // 清理空字符串
        foreach ($data as $key => $value) {
            if ($value === '') {
                $data[$key] = null;
            }
        }

        // 处理数字字段
        $numericFields = ['click', 'sort', 'price'];
        foreach ($numericFields as $field) {
            if (isset($data[$field])) {
                $data[$field] = is_numeric($data[$field]) ? (int)$data[$field] : 0;
            }
        }

        return $data;
    }

    // -------------------------------------------------------------------------
    // 批量删除
    // -------------------------------------------------------------------------

    public function batchDelete(array $ids): array
    {
        Db::startTrans();
        try {
            $count = RyArticleModel::whereIn('id', $ids)->delete();

            if ($count === 0) {
                Db::rollback();
                return ['code' => 400, 'msg' => "未找到可删除的数据"];
            }

            Db::commit();
            return ['code' => 200, 'msg' => "删除成功", 'count' => $count];
            
        } catch (\Throwable $e) {
            Db::rollback();
            Log::error("文章批量删除失败: {$e->getMessage()}");
            
            return [
                'code' => 500,
                'msg' => '删除操作失败，请稍后重试'
            ];
        }
    }

    // -------------------------------------------------------------------------
    // 获取文章统计
    // -------------------------------------------------------------------------

    public function getArticleStats(int $cateId = 0): array
    {
        $query = RyArticleModel::query();
        
        if ($cateId > 0) {
            $query->where('cateid', $cateId);
        }

        return [
            'total' => $query->count(),
            'published' => $query->where('status', 1)->count(),
            'draft' => $query->where('status', 0)->count(),
            'today' => $query->whereDay('create_time', date('d'))->count()
        ];
    }
}