<?php
// install.php - 主安装程序文件（只处理逻辑，不输出HTML）
// -----------------------------------------------------------------------------
/*!
 * RoyalCMS 若伊智能网站构建系统 - 一键安装程序
 * 
 * @name      RoyalCMS 一键安装程序
 * @version   2.0.0
 * @author    RoyalCMS Team
 * @copyright Copyright (c) 2018-2024 RoyalCMS.keeyoung.cn All rights reserved.
 * @license   MIT License
 * @homepage  https://www.royalcms.com.cn
 */
// -----------------------------------------------------------------------------

declare(strict_types=1);

// 错误报告设置
error_reporting(E_ALL);
ini_set('display_errors', '1');
date_default_timezone_set('Asia/Shanghai');

// 定义常量
define('ROYAL_INSTALL', true);
define('ROYAL_VERSION', '2.0.0');
define('ROYAL_ROOT', dirname(__DIR__)); // public的父目录（系统根目录）
define('INSTALL_DIR', __DIR__); // public/install/
define('SYSTEM_ROOT', dirname(ROYAL_ROOT)); // 系统根目录
define('RUNTIME_DIR', SYSTEM_ROOT . '/runtime'); // 系统目录下的runtime

// 临时定义env函数，用于读取配置文件
if (!function_exists('env')) {
    function env($name, $default = null) {
        return $default;
    }
}

class RoyalInstaller
{
    private $config = [];
    private $errors = [];
    
    public function __construct()
    {
        session_start();
        $this->loadDatabaseConfig();
    }
    
    // 显示已安装提示
    private function showInstalledMessage()
    {
        echo '<!DOCTYPE html>
        <html lang="zh-CN">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>系统已安装 - RoyalCMS</title>
            <style>
                * { margin: 0; padding: 0; box-sizing: border-box; }
                body { 
                    font-family: "Microsoft YaHei", Arial, sans-serif; 
                    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); 
                    height: 100vh; 
                    display: flex; 
                    align-items: center; 
                    justify-content: center; 
                    margin: 0; 
                }
                .container { 
                    background: white; 
                    padding: 40px; 
                    border-radius: 10px; 
                    text-align: center; 
                    box-shadow: 0 10px 30px rgba(0,0,0,0.2); 
                    max-width: 640px;
                    margin: 20px;
                }
                h1 { color: #2c3e50; margin-bottom: 20px; }
                p { color: #7f8c8d; margin-bottom: 10px; line-height: 1.6; }
                .btn { 
                    display: inline-block; 
                    padding: 12px 30px; 
                    background: #3498db; 
                    color: white; 
                    text-decoration: none; 
                    border-radius: 5px; 
                    margin: 10px; 
                    transition: all 0.3s; 
                    border: none;
                    cursor: pointer;
                    font-size: 16px;
                }
                .btn:hover { 
                    background: #2980b9; 
                    transform: translateY(-2px); 
                    box-shadow: 0 5px 15px rgba(0,0,0,0.2);
                }
                .info-box {
                    background: #e8f5e8;
                    padding: 20px;
                    border-radius: 5px;
                    margin: 20px 0;
                    text-align: left;
                }
                .info-box h3 {
                    color: #27ae60;
                    margin-bottom: 10px;
                }
                    
                .danger-alert {
    margin: 0 0 25px 0;
    border: 2px solid #dc3545;
    border-radius: 8px;
    background: #fff5f5;
    overflow: hidden;
}

.alert-title {
    background: #dc3545;
    color: white;
    padding: 12px 15px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.alert-title h3 {
    margin: 0;
    font-size: 16px;
    font-weight: 600;
}

.alert-content {
    padding: 15px;
}

.alert-content > p {
    margin: 0 0 12px 0;
    color: #721c24;
    font-size: 14px;
}

.security-list {
    list-style: none;
    margin: 0;
    padding: 0;
}

.security-list li {
    display: flex;
    align-items: flex-start;
    padding: 10px 0;
    border-bottom: 1px solid #f8d7da;
}

.security-list li:last-child {
    border-bottom: none;
}

.security-list .step {
    width: 22px;
    height: 22px;
    background: #dc3545;
    color: white;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 12px;
    font-weight: bold;
    margin-right: 10px;
    flex-shrink: 0;
    margin-top: 1px;
}

.security-list strong {
    flex: 0 0 180px;
    color: #721c24;
    font-size: 14px;
    font-weight: 600;
}

.security-list .path {
    flex: 1; 
    font-size: 13px;
    color: #495057;
    background: white;
    padding: 3px 8px;
    border-radius: 3px;
    border: 1px solid #f1aeb5;
}
    
            </style>
        </head>
        <body>
            <div class="container">
                <h1>RoyalCMS 系统已安装</h1>
                <p>检测到系统已经安装完成，无需重复安装。</p>
                
                <div class="info-box">
                    <h3>登录信息：</h3>
                    <p>后台地址: /royalcms/website/login/index</p>
                    <p>测试账号: demo</p>
                    <p>测试密码: 123456</p>
                </div>
                <div class="danger-alert">
                <div class="alert-title"> 
                    <h3>重要！请按顺序完成以下操作，否则系统可能被他人重新安装</h3>
                </div>
                <div class="alert-content"> 
                    <ul class="security-list">
                        <li>
                            <span class="step">1</span>
                            <strong>删除整个 install 目录</strong>
                            <span class="path">/public/install/</span>
                        </li>
                        <li>
                            <span class="step">2</span>
                            <strong>删除网站根目录的 index.php</strong> 
                            <span class="path">目录public/index.php</span>
                        </li>
                        <li>
                            <span class="step">3</span>
                            <strong>将 index_bak.php 重命名为 index.php</strong>
                            <span class="path">/public/index_bak.php → /public/index.php</span>
                        </li>
                        <li>
                            <span class="step">4</span>
                            <strong>立即登录后台修改密码</strong>
                            <span class="path">使用 demo/123456 登录后立即修改</span>
                        </li>
                    </ul>
                </div>
            </div>
            
                <div>
                    <a href="/" class="btn">访问网站首页</a>
                    <a href="/royalcms/website/login/index" class="btn" target="_blank">进入后台管理</a>
                </div>
                
                <div style="margin-top: 20px; font-size: 14px; color: #95a5a6;">
                    <p>如需重新安装，请先删除 ' . RUNTIME_DIR . '/install.lock 文件</p>
                </div>
            </div>
        </body>
        </html>';
        exit;
    }
    
    // 检查是否已安装
    private function checkInstalled()
    {
        if (file_exists(RUNTIME_DIR . '/install.lock')) {
            $this->showInstalledMessage();
            return true;
        }
        return false;
    }
    
    // 加载数据库配置
    private function loadDatabaseConfig()
    {
        $configFile = SYSTEM_ROOT . '/config/database.php';
        
        if (!file_exists($configFile)) {
            // 使用默认配置
            $this->config = [
                'db_host' => '127.0.0.1',
                'db_port' => '3306',
                'db_user' => 'root',
                'db_pass' => '',
                'db_name' => 'royalcms',
                'db_charset' => 'utf8'
            ];
            return;
        }
        
        try {
            // 临时定义env函数，防止配置文件中的env()调用报错
            if (!function_exists('env')) {
                function env($name, $default = null) {
                    return $default;
                }
            }
            
            // 屏蔽可能的环境变量读取错误
            $configArray = @require $configFile;
            
            if (empty($configArray) || !is_array($configArray)) {
                throw new Exception('数据库配置文件格式错误');
            }
            
            // 读取ThinkPHP格式的配置
            if (isset($configArray['connections']['mysql'])) {
                $mysqlConfig = $configArray['connections']['mysql'];
                
                $this->config = [
                    'db_host' => $mysqlConfig['hostname'] ?? ($mysqlConfig['hostname'] ?? '127.0.0.1'),
                    'db_port' => $mysqlConfig['hostport'] ?? ($mysqlConfig['hostport'] ?? '3306'),
                    'db_user' => $mysqlConfig['username'] ?? ($mysqlConfig['username'] ?? 'root'),
                    'db_pass' => $mysqlConfig['password'] ?? ($mysqlConfig['password'] ?? ''),
                    'db_name' => $mysqlConfig['database'] ?? ($mysqlConfig['database'] ?? 'royalcms'),
                    'db_charset' => $mysqlConfig['charset'] ?? ($mysqlConfig['charset'] ?? 'utf8')
                ];
            } else {
                // 如果没有connections配置，尝试直接读取
                $this->config = [
                    'db_host' => $configArray['hostname'] ?? '127.0.0.1',
                    'db_port' => $configArray['hostport'] ?? '3306',
                    'db_user' => $configArray['username'] ?? 'root',
                    'db_pass' => $configArray['password'] ?? '',
                    'db_name' => $configArray['database'] ?? 'royalcms',
                    'db_charset' => $configArray['charset'] ?? 'utf8'
                ];
            }
            
            // 验证配置是否完整
            if (empty($this->config['db_host']) || empty($this->config['db_user']) || empty($this->config['db_name'])) {
                throw new Exception('数据库配置不完整');
            }
            
        } catch (Exception $e) {
            // 配置读取失败，使用默认值
            $this->config = [
                'db_host' => '127.0.0.1',
                'db_port' => '3306',
                'db_user' => 'root',
                'db_pass' => '',
                'db_name' => 'royalcms',
                'db_charset' => 'utf8'
            ];
            
            error_log('数据库配置读取失败: ' . $e->getMessage());
        }
    }
    
    public function run()
    {
        // 检查是否已安装
        if ($this->checkInstalled()) {
            return;
        }
        
        $step = $_GET['step'] ?? 'welcome';
        $action = $_GET['action'] ?? '';
        
        // 处理API请求
        if ($action) {
            $this->handleApi($action);
            return;
        }
        
        // 加载对应步骤的模板
        $this->loadStep($step);
    }
    
    private function loadStep($step)
    {
        $templateFile = INSTALL_DIR . '/templates/' . $step . '.html';
        
        if (!file_exists($templateFile)) {
            die('模板文件不存在: ' . $templateFile);
        }
        
        // 准备模板数据
        $data = $this->prepareTemplateData($step);
        
        // 输出HTML框架并包含模板
        $this->outputTemplate($templateFile, $data, $this->getPageTitle($step));
    }
    
    private function prepareTemplateData($step)
    {
        $data = [
            'version' => ROYAL_VERSION,
            'base_url' => $this->getBaseUrl(),
            'config' => $this->config // 传递数据库配置到模板
        ];
        
        switch ($step) {
            case 'welcome':
                break;
                
            case 'check':
                $data['checks'] = $this->performChecks();
                $data['all_passed'] = true;
                foreach ($data['checks'] as $check) {
                    if (!$check['status']) {
                        $data['all_passed'] = false;
                        break;
                    }
                }
                break;
                
            case 'complete':
                $data['site_config'] = [
                    'site_name' => 'RoyalCMS',
                    'admin_user' => 'demo'
                ];
                $data['install_time'] = date('Y-m-d H:i:s');
                
                // 添加安装成功后的额外信息
                $data['success_info'] = [
                    'database' => $this->config['db_name'],
                    'host' => $this->config['db_host'] . ':' . $this->config['db_port'],
                    'lock_file' => RUNTIME_DIR . '/install.lock'
                ];
                break;
                
            case 'error':
                $data['error_message'] = $_SESSION['install_error'] ?? '未知错误';
                unset($_SESSION['install_error']);
                break;
        }
        
        return $data;
    }
    
    private function outputTemplate($templateFile, $data, $title)
    {
        extract($data);
        
        // 输出HTML框架
        echo '<!DOCTYPE html>
        <html lang="zh-CN">
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
            <title>' . $title . '</title>
            <link rel="stylesheet" href="' . $this->getBaseUrl() . 'assets/css/css.css">
        </head>
        <body>
            <div class="install-container">
                <div class="install-header">
                    <h1>RoyalCMS 安装程序</h1>
                    <div class="version">版本 v' . ROYAL_VERSION . '</div>
                </div>';
        
        // 包含模板文件
        include $templateFile;
        
        // 输出JavaScript
        echo '
            </div>
            <script>
                // 全局变量
                window.installer = {
                    baseUrl: "' . $this->getBaseUrl() . '",
                    currentStep: "' . ($_GET['step'] ?? 'welcome') . '"
                };
                
                // 安装函数
                function startInstallation() {
                    const progress = document.getElementById("installProgress");
                    const text = document.getElementById("progressText");
                    const logContent = document.getElementById("logContent");
                    
                    function addLog(message, type = "info") {
                        const timestamp = new Date().toLocaleTimeString();
                        const logEntry = document.createElement("div");
                        logEntry.className = "log-entry";
                        logEntry.innerHTML = `<span class="log-time">[${timestamp}]</span> ${message}`;
                        logContent.appendChild(logEntry);
                        logContent.scrollTop = logContent.scrollHeight;
                    }
                    
                    function updateProgress(percent, message) {
                        if (progress) progress.style.width = percent + "%";
                        if (text) text.textContent = message;
                        addLog(message);
                    }
                    
                    // 模拟安装过程
                    updateProgress(10, "准备安装环境...");
                    
                    setTimeout(() => {
                        updateProgress(30, "正在连接数据库...");
                        
                        // 实际执行安装
                        fetch("install.php?action=perform-install")
                            .then(response => response.json())
                            .then(data => {
                                if (data.success) {
                                    updateProgress(80, "数据库创建完成...");
                                    addLog("✅ 数据库创建成功", "success");
                                    
                                    setTimeout(() => {
                                        updateProgress(90, "导入数据表中...");
                                        addLog("📊 正在导入数据表", "info");
                                        
                                        setTimeout(() => {
                                            updateProgress(100, "安装完成！");
                                            addLog("🎉 系统安装完成！", "success");
                                            
                                            // 显示成功信息，然后跳转
                                            setTimeout(() => {
                                                window.location.href = "?step=complete";
                                            }, 2000);
                                        }, 1500);
                                    }, 1000);
                                } else {
                                    updateProgress(0, "安装失败");
                                    addLog("❌ " + data.message, "error");
                                    
                                    setTimeout(() => {
                                        window.location.href = "?step=error";
                                    }, 2000);
                                }
                            })
                            .catch(error => {
                                updateProgress(0, "安装请求失败");
                                addLog("❌ " + error.message, "error");
                                
                                setTimeout(() => {
                                    window.location.href = "?step=error";
                                }, 2000);
                            });
                            
                    }, 1000);
                }
                
                // 自动开始安装
                document.addEventListener("DOMContentLoaded", function() {
                    if (window.installer.currentStep === "installing") {
                        console.log("开始安装过程...");
                        startInstallation();
                    }
                });
                
                // 检查安全操作完成状态（仅在complete页面）
                if (window.installer.currentStep === "complete") {
                    console.log("当前在完成页面");
                    document.addEventListener("DOMContentLoaded", function() {
                        // 检查是否已确认安全操作
                        const isSecurityDone = localStorage.getItem("royalcms_safe_check") === "completed";
                        const loginBtn = document.getElementById("loginBtn");
                        const homeBtn = document.getElementById("homeBtn");
                        const unlockNotice = document.getElementById("unlockNotice");
                        
                        console.log("安全操作状态:", isSecurityDone);
                        
                        if (isSecurityDone) {
                            // 已确认，解锁按钮
                            console.log("已确认安全操作，解锁按钮");
                            if (loginBtn) {
                                loginBtn.style.pointerEvents = "auto";
                                loginBtn.style.opacity = "1";
                            }
                            if (homeBtn) {
                                homeBtn.style.pointerEvents = "auto";
                                homeBtn.style.opacity = "1";
                            }
                            if (unlockNotice) {
                                unlockNotice.style.display = "none";
                            }
                        } else {
                            // 未确认，锁定按钮
                            console.log("未确认安全操作，锁定按钮");
                            if (loginBtn) {
                                loginBtn.style.pointerEvents = "none";
                                loginBtn.style.opacity = "0.6";
                            }
                            if (homeBtn) {
                                homeBtn.style.pointerEvents = "none";
                                homeBtn.style.opacity = "0.6";
                            }
                        }
                    });
                }
                
                // 完成安全操作确认函数（在complete.html模板中调用）
                function completeSecurityCheck() {
                    if (confirm("您确认已完成以下所有安全操作吗？\\n\\n1. 已删除整个 /public/install 目录\\n2. 已删除网站根目录的 index.php\\n3. 已将 index_bak.php 改名为 index.php\\n\\n如果未完成，请先完成这些操作以确保系统安全！")) {
                        localStorage.setItem("royalcms_safe_check", "completed");
                        alert("✅ 安全确认完成！现在可以正常访问网站。\\n\\n重要提醒：请立即登录后台修改默认密码！");
                        
                        // 刷新页面以更新按钮状态
                        window.location.reload();
                    }
                }
                
                // 测试跳转函数
                function testCompletePage() {
                    window.location.href = "?step=complete";
                }
            </script>
        </body>
        </html>';
        exit;
    }
    
    // 处理API请求
    private function handleApi($action)
    {
        header('Content-Type: application/json; charset=utf-8');
        
        switch ($action) {
            case 'perform-install':
                $result = $this->performInstallation();
                echo json_encode($result);
                if (!$result['success']) {
                    $_SESSION['install_error'] = $result['message'];
                }
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => '未知操作']);
        }
        exit;
    }
    
    // 执行环境检查
    private function performChecks()
    {
        $mysqlCheck = $this->checkMySQLConnection();
        
        // 显示当前使用的数据库配置
        $configInfo = "主机: {$this->config['db_host']}:{$this->config['db_port']}, 用户: {$this->config['db_user']}, 数据库: {$this->config['db_name']}";
        
        return [
            'php_version' => [
                'name' => 'PHP版本',
                'required' => '7.4+',
                'current' => PHP_VERSION,
                'status' => version_compare(PHP_VERSION, '7.4.0') >= 0
            ],
            'pdo_mysql' => [
                'name' => 'PDO MySQL扩展',
                'required' => '必须',
                'current' => extension_loaded('pdo_mysql') ? '已安装' : '未安装',
                'status' => extension_loaded('pdo_mysql')
            ],
            'sql_file' => [
                'name' => 'SQL文件',
                'required' => '存在且可读',
                'current' => file_exists(INSTALL_DIR . '/royalcms.sql') ? '存在' : '不存在',
                'status' => file_exists(INSTALL_DIR . '/royalcms.sql') && is_readable(INSTALL_DIR . '/royalcms.sql')
            ],
            'config_file' => [
                'name' => '数据库配置文件',
                'required' => '存在且可读',
                'current' => file_exists(SYSTEM_ROOT . '/config/database.php') ? '存在' : '不存在',
                'status' => file_exists(SYSTEM_ROOT . '/config/database.php') && is_readable(SYSTEM_ROOT . '/config/database.php')
            ],
            'mysql_connection' => [
                'name' => 'MySQL连接',
                'required' => '可连接',
                'current' => $mysqlCheck['current'],
                'status' => $mysqlCheck['status'],
                'message' => $mysqlCheck['message'] . ($mysqlCheck['status'] ? '' : ' (' . $configInfo . ')')
            ]
        ];
    }
    
    // 检查MySQL连接
    private function checkMySQLConnection()
    {
        try {
            // 从配置文件读取的连接信息
            $dsn = "mysql:host={$this->config['db_host']};port={$this->config['db_port']};charset={$this->config['db_charset']}";
            $pdo = new PDO($dsn, $this->config['db_user'], $this->config['db_pass']);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            return [
                'status' => true,
                'message' => '数据库连接成功',
                'current' => '可连接'
            ];
            
        } catch (PDOException $e) {
            $errorInfo = "连接信息: {$this->config['db_host']}:{$this->config['db_port']}, 用户: {$this->config['db_user']}, 数据库: {$this->config['db_name']}";
            return [
                'status' => false,
                'message' => '数据库连接失败: ' . $e->getMessage() . ' ' . $errorInfo,
                'current' => '连接失败'
            ];
        } catch (Exception $e) {
            $errorInfo = "连接信息: {$this->config['db_host']}:{$this->config['db_port']}, 用户: {$this->config['db_user']}, 数据库: {$this->config['db_name']}";
            return [
                'status' => false,
                'message' => '数据库连接异常: ' . $e->getMessage() . ' ' . $errorInfo,
                'current' => '连接异常'
            ];
        }
    }
    
    // 执行安装
    private function performInstallation()
    {
        try {
            // 检查数据库配置
            $dbCheck = $this->checkMySQLConnection();
            if (!$dbCheck['status']) {
                return ['success' => false, 'message' => '数据库连接失败: ' . $dbCheck['message']];
            }
            
            // 检查SQL文件
            $sqlFile = INSTALL_DIR . '/royalcms.sql';
            if (!file_exists($sqlFile)) {
                return ['success' => false, 'message' => 'SQL文件不存在'];
            }
            
            // 1. 创建数据库
            $this->createDatabase();
            
            // 2. 导入SQL文件
            $this->importSQL();
            
            // 3. 创建安装锁文件（在系统目录的runtime下）
            $this->createLockFile();
            
            return ['success' => true, 'message' => '安装成功，数据库创建并数据导入完成！'];
            
        } catch (Exception $e) {
            return ['success' => false, 'message' => '安装失败: ' . $e->getMessage()];
        }
    }
    
    // 创建数据库
    private function createDatabase()
    {
        try {
            $dsn = "mysql:host={$this->config['db_host']};port={$this->config['db_port']};charset={$this->config['db_charset']}";
            $pdo = new PDO($dsn, $this->config['db_user'], $this->config['db_pass']);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            // 创建数据库（如果不存在）
            $charset = str_replace('-', '', $this->config['db_charset']); // utf8mb4 -> utf8mb4
            $sql = "CREATE DATABASE IF NOT EXISTS `{$this->config['db_name']}` 
                    CHARACTER SET {$charset} COLLATE {$charset}_unicode_ci";
            $pdo->exec($sql);
            
        } catch (Exception $e) {
            throw new Exception('数据库创建失败: ' . $e->getMessage());
        }
    }
    
    // 导入SQL文件
    private function importSQL()
    {
        $sqlFile = INSTALL_DIR . '/royalcms.sql';
        
        if (!file_exists($sqlFile)) {
            throw new Exception('SQL文件不存在');
        }
        
        try {
            $dsn = "mysql:host={$this->config['db_host']};port={$this->config['db_port']};dbname={$this->config['db_name']};charset={$this->config['db_charset']}";
            $pdo = new PDO($dsn, $this->config['db_user'], $this->config['db_pass']);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            // 读取SQL文件
            $sqlContent = file_get_contents($sqlFile);
            
            // 分割SQL语句
            $sqlStatements = array_filter(array_map('trim', explode(';', $sqlContent)));
            
            // 执行每个SQL语句
            foreach ($sqlStatements as $statement) {
                if (!empty($statement)) {
                    try {
                        $pdo->exec($statement);
                    } catch (Exception $e) {
                        // 记录错误但继续执行
                        error_log('SQL执行错误: ' . $e->getMessage() . ' SQL: ' . substr($statement, 0, 100));
                    }
                }
            }
            
        } catch (Exception $e) {
            throw new Exception('SQL导入失败: ' . $e->getMessage());
        }
    }
    
    // 创建安装锁文件（在系统目录的runtime下）
    private function createLockFile()
    {
        // 确保runtime目录存在
        if (!is_dir(RUNTIME_DIR)) {
            if (!mkdir(RUNTIME_DIR, 0755, true)) {
                throw new Exception('无法创建runtime目录: ' . RUNTIME_DIR);
            }
        }
        
        $lockContent = "RoyalCMS Installation Lock\n";
        $lockContent .= "Version: " . ROYAL_VERSION . "\n";
        $lockContent .= "Database: " . $this->config['db_name'] . "\n";
        $lockContent .= "InstallTime: " . date('Y-m-d H:i:s') . "\n";
        $lockContent .= "DemoAccount: demo/123456\n";
        $lockContent .= "AdminURL: /royalcms/website/login/index\n";
        
        if (!file_put_contents(RUNTIME_DIR . '/install.lock', $lockContent)) {
            throw new Exception('无法创建安装锁文件: ' . RUNTIME_DIR . '/install.lock');
        }
    }
    
    // 获取页面标题
    private function getPageTitle($step)
    {
        $titles = [
            'welcome' => '欢迎安装 RoyalCMS',
            'check' => '环境检测',
            'installing' => '安装中',
            'complete' => '安装完成',
            'error' => '安装错误'
        ];
        
        return isset($titles[$step]) ? $titles[$step] : 'RoyalCMS 安装程序';
    }
    
    // 获取基础URL
    private function getBaseUrl()
    {
        $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $script = $_SERVER['SCRIPT_NAME'] ?? '';
        $installDir = dirname($script);
        
        if ($installDir === '/' || $installDir === '\\') {
            return $protocol . '://' . $host . '/';
        }
        
        return $protocol . '://' . $host . $installDir . '/';
    }
}

// 运行安装程序
try {
    $installer = new RoyalInstaller();
    $installer->run();
} catch (Exception $e) {
    die('安装程序初始化失败: ' . $e->getMessage());
}