// RoyalCMS Mobile Optimized - Navigation and Modal Functions
$(document).ready(function () {
    royalInitNavigation();
    royalInitModals();
    royalInitScrollEffects();
    royalInitFloatingWarning();
    royalInitTouchEvents();

    // Initialize AOS scroll animations
    if (typeof AOS !== 'undefined') {
        AOS.init({
            duration: 300,
            once: true,
            offset: 50,
            delay: 10,
            easing: 'ease-in-out'
        });
    }
});

// Navigation - Completely rewritten for mobile optimization
function royalInitNavigation() {
    const $header = $('.royal-header');
    const $navMenu = $('.royal-nav-menu');
    const $mobileToggle = $('.royal-mobile-menu-toggle');
    const $mobileOverlay = $('.royal-mobile-overlay');
    const $navItems = $('.royal-nav-menu > li');
    const $dropdownItems = $('.royal-has-dropdown');

    // Mobile menu toggle
    function toggleMobileMenu() {
        $mobileToggle.toggleClass('royal-active');
        $navMenu.toggleClass('royal-active');
        $mobileOverlay.toggleClass('royal-active');
        $('body').toggleClass('royal-mobile-menu-open');

        // Close all dropdowns when main menu closes
        if (!$navMenu.hasClass('royal-active')) {
            $dropdownItems.removeClass('royal-active');
            $('.royal-dropdown-menu').slideUp(200);
        }
    }

    // Mobile menu button click
    $mobileToggle.on('click', function (e) {
        e.stopPropagation();
        toggleMobileMenu();
    });

    // Overlay click to close menu
    $mobileOverlay.on('click', function () {
        toggleMobileMenu();
    });

    // Mobile dropdown handling
    $('.royal-has-dropdown > a').on('click', function (e) {
        if ($(window).width() <= 768) {
            e.preventDefault();
            const $parent = $(this).parent();
            const $dropdown = $parent.find('.royal-dropdown-menu');
            const $icon = $(this).find('.fa-chevron-down');

            // Close other dropdowns
            $dropdownItems.not($parent).removeClass('royal-active');
            $('.royal-dropdown-menu').not($dropdown).slideUp(200);
            $('.royal-has-dropdown > a .fa-chevron-down').not($icon).removeClass('royal-rotated');

            // Toggle current dropdown
            $parent.toggleClass('royal-active');
            $dropdown.slideToggle(200);
            $icon.toggleClass('royal-rotated');
        }
    });

    // Desktop dropdown
    $navItems.hover(
        function () {
            if ($(window).width() > 768) {
                $(this).addClass('royal-hover');
                $(this).find('.royal-dropdown-menu').stop().slideDown(200);
            }
        },
        function () {
            if ($(window).width() > 768) {
                $(this).removeClass('royal-hover');
                $(this).find('.royal-dropdown-menu').stop().slideUp(150);
            }
        }
    );

    // Header scroll effect
    $(window).on('scroll', royalThrottle(function () {
        const scrollY = $(this).scrollTop();
        if (scrollY > 100) {
            $header.addClass('royal-scrolled');
        } else {
            $header.removeClass('royal-scrolled');
        }
    }, 100));

    // Reset menu state on window resize
    $(window).on('resize', royalThrottle(function () {
        if ($(window).width() > 768) {
            // Desktop: close mobile menu states
            $mobileToggle.removeClass('royal-active');
            $navMenu.removeClass('royal-active');
            $mobileOverlay.removeClass('royal-active');
            $('body').removeClass('royal-mobile-menu-open');
            $dropdownItems.removeClass('royal-active');
            $('.royal-dropdown-menu').removeAttr('style');
        } else {
            // Mobile: ensure desktop hover effects are cleared
            $navItems.removeClass('royal-hover');
            $('.royal-dropdown-menu').removeAttr('style');
        }
    }, 250));
}

// Modal functions optimization
function royalInitModals() {
    let currentModal = null;

 

    // Close on ESC key
    $(document).on('keydown', function (e) {
        if (e.keyCode === 27 && currentModal) {
            royalHideModal();
        }
    });

    // Download confirmation
    $('#royal-confirmDownload').on('click', function () {
        royalShowNotification('Thank you for your support! Download link activated', 'success');
        royalHideModal();
    });

 
 
}

// Scroll effects optimization
function royalInitScrollEffects() {
    // Back to top button
    $('.royal-fixed-btn.royal-top').on('click', function () {
        $('html, body').animate({ scrollTop: 0 }, 800);
    });

    // Show/hide back to top button
    $(window).on('scroll', royalThrottle(function () {
        const $topBtn = $('.royal-fixed-btn.royal-top');
        const scrollTop = $(this).scrollTop();

        if (scrollTop > 500) {
            $topBtn.addClass('royal-visible');
        } else {
            $topBtn.removeClass('royal-visible');
        }
    }, 150));

    // Smooth scroll - optimized for mobile
    $('.royal-nav-menu a[href*="#"]').on('click', function (e) {
        const targetId = $(this).attr('href');

        if (targetId.startsWith('#') && $(targetId).length) {
            e.preventDefault();

            // Mobile: close menu
            if ($(window).width() <= 768) {
                $('.royal-mobile-menu-toggle').trigger('click');
            }

            const $target = $(targetId);
            const offsetTop = $target.offset().top - 80;

            $('html, body').animate({
                scrollTop: offsetTop
            }, 800);
        }
    });
}

// Floating warning function
function royalInitFloatingWarning() {
    const $royalWarning = $('#royal-floatingWarning');
    if (!$royalWarning.length) return;

    // Close warning
    $('.royal-close-warning').on('click', function () {
        $royalWarning.slideUp(300, function () {
            $(this).addClass('royal-hidden');
        });
        royalSetCookie('royalWarningClosed', 'true', 1);
    });

    // Auto hide
    const warningClosed = royalGetCookie('royalWarningClosed');
    if (warningClosed === 'true') {
        $royalWarning.hide().addClass('royal-hidden');
    } else {
        setTimeout(function () {
            if (!$royalWarning.hasClass('royal-hidden')) {
                $royalWarning.slideUp(300, function () {
                    $(this).addClass('royal-hidden');
                });
                royalSetCookie('royalWarningClosed', 'true', 1);
            }
        }, 10000);
    }
}

// Mobile touch events support
function royalInitTouchEvents() {
    let startX, startY;

    // Swipe to close menu
    document.addEventListener('touchstart', function (e) {
        startX = e.touches[0].clientX;
        startY = e.touches[0].clientY;
    }, false);

    document.addEventListener('touchend', function (e) {
        if (!startX || !startY) return;

        const endX = e.changedTouches[0].clientX;
        const endY = e.changedTouches[0].clientY;
        const diffX = startX - endX;
        const diffY = startY - endY;

        // If significant left to right swipe with minimal vertical movement
        if (Math.abs(diffX) > 50 && Math.abs(diffY) < 30 && diffX > 0) {
            // Close mobile menu
            if ($('.royal-nav-menu').hasClass('royal-active')) {
                $('.royal-mobile-menu-toggle').trigger('click');
            }
        }

        startX = null;
        startY = null;
    }, false);
}

// Utility functions
function royalShowNotification(message, type = 'info') {
    const $notification = $(`
        <div class="royal-notification royal-${type}">
            <div class="royal-notification-content">
                <i class="fas fa-${type === 'success' ? 'check' : type === 'error' ? 'exclamation' : 'info'}-circle"></i>
                <span>${message}</span>
            </div>
        </div>
    `);

    $('body').append($notification);

    setTimeout(() => {
        $notification.addClass('royal-show');
    }, 100);

    setTimeout(() => {
        $notification.removeClass('royal-show');
        setTimeout(() => {
            $notification.remove();
        }, 300);
    }, 3000);
}

function royalCopyToClipboard(text) {
    if (navigator.clipboard && window.isSecureContext) {
        navigator.clipboard.writeText(text);
    } else {
        const textArea = document.createElement('textarea');
        textArea.value = text;
        textArea.style.position = 'fixed';
        textArea.style.opacity = '0';
        document.body.appendChild(textArea);
        textArea.select();
        document.execCommand('copy');
        document.body.removeChild(textArea);
    }
}

function royalThrottle(func, limit) {
    let inThrottle;
    return function () {
        const args = arguments;
        const context = this;
        if (!inThrottle) {
            func.apply(context, args);
            inThrottle = true;
            setTimeout(() => inThrottle = false, limit);
        }
    };
}

function royalSetCookie(name, value, days) {
    const d = new Date();
    d.setTime(d.getTime() + (days * 24 * 60 * 60 * 1000));
    const expires = "expires=" + d.toUTCString();
    document.cookie = name + "=" + value + ";" + expires + ";path=/";
}

function royalGetCookie(name) {
    const nameEQ = name + "=";
    const ca = document.cookie.split(';');
    for (let i = 0; i < ca.length; i++) {
        let c = ca[i].trim();
        if (c.indexOf(nameEQ) === 0) return c.substring(nameEQ.length);
    }
    return null;
}